<?php

namespace FreePBX\Modules;

use FreePBX\Freepbx_conf;

class Zruchnaio extends \FreePBX_Helpers implements \BMO
{
    /** @var \FreePBX $FreePBX */
    protected $FreePBX;

    /** @var \DB */
    protected $db;

    /** @var Freepbx_conf */
    private $freePbxConf;

    /**
     * @var \AGI_AsteriskManager
     */
    private $astman;

    /**
     * @param \FreePBX $freepbx
     * @throws \Exception
     * @throws \BadMethodCallException
     */
    public function __construct($freepbx = null)
    {
        parent::__construct($freepbx);

        if (null === $freepbx) {
            throw new \BadMethodCallException('Not given a FreePBX Object');
        }

        $this->FreePBX = $freepbx;
        $this->db = $freepbx->Database;
        $this->freePbxConf = Freepbx_conf::create();
        $this->astman = $freepbx->astman;
    }

    /**
     * Set Priority for Dialplan Hooking
     * Core sits at a priority of 600
     * @method myDialplanHooks
     *
     * @return string        Priority
     */
    public static function myDialplanHooks()
    {
        return 900;
    }

    public function install()
    {
        // global $amp_conf;

        $this->setMultiConfig([
            'freepbx_module_secret' => $this->createSecret(),
            'freepbx_base_uri' => '',
            'records_extension' => 'wav',
            'freepbx_callback_extension' => '',
            'client_name' => '',
        ]);

        // $htdocs_dest = $amp_conf['AMPWEBROOT'];
        // $bin_dest    = isset($amp_conf['AMPBIN']) ? $amp_conf['AMPBIN'] : '/var/lib/asterisk/bin';
        // $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $this->addAgiBinFile('accountuniqe');
        $this->addAgiBinFile('accountuniqeout');

        $this->freePbxConf->set_conf_values(array(
            'DIAL_OPTIONS' => 'HhTtrU(callanswer)',
            'TRUNK_OPTIONS' => 'TM(outdial)',
        ));
        $this->freePbxConf->commit_conf_settings();
    }

    /**
     * @return string
     */
    private function createSecret()
    {
        return base64_encode(openssl_random_pseudo_bytes(16));
    }

    public function uninstall()
    {
        $this->removeAgiBinFile('accountuniqe');
        $this->removeAgiBinFile('accountuniqeout');
    }

    private function addAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $srcFilePath = __DIR__.'/agi/'.$filename.'.sh';
        $dstFilePath = $agibin_dest.'/'.$filename.'.sh';

        if (file_exists($dstFilePath)) {
            return;
        }

        symlink($srcFilePath, $dstFilePath);

        if (file_exists($dstFilePath) && !is_executable($dstFilePath)) {
            if (false !== $dstFileStat = stat($dstFilePath)) {
                $dstFileMode = $dstFileStat['mode'] | 0111;
                chmod($dstFilePath, $dstFileMode);
            }
        }
    }

    private function removeAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $dstFilePath = $agibin_dest.'/'.$filename.'.sh';

        if (file_exists($dstFilePath)) {
            unlink($dstFilePath);
        }
    }

    public function backup()
    {
    }

    public function restore($backup)
    {
    }

    /**
     * Hook into Dialplan (extensions_additional.conf)
     * @method doDialplanHook
     *
     * @param \extensions $ext The Extensions Class https://wiki.freepbx.org/pages/viewpage.action?pageId=98701336
     * @param string $engine Always Asterisk, Legacy
     * @param string $priority Priority
     */
    public function doDialplanHook($ext, $engine, $priority)
    {
        $this->registerGlobalVariables($ext);

        $this->addFromPstnCustomContext($ext);
        $this->addNewExtDidContext($ext);
        $this->addZamenaContext($ext);
        $this->addNumberadddigitContext($ext);
        $this->addMisscallContext($ext);
        $this->addMisscalldbdelContext($ext);
        $this->addMacroOutdialContext($ext);
        $this->addCallanswerContext($ext);
        $this->addCallansweContext($ext);
        $this->addInCallEndContext($ext);
        $this->addInCallEndDbdelContext($ext);
        $this->addOutnewCdr($ext);

        $this->modifyMacroHangupcallContext($ext);
        $this->modifyMacroDialoutTrunk($ext);
        $this->modifyMacroDialOneContext($ext);
        $this->modifyMacroDialContext($ext);
    }

    private function registerGlobalVariables(\extensions $ext)
    {
        if ('' !== $clientName = $this->getClientName()) {
            $ext->addGlobal('ZRUCHNA_CLIENT_NAME', $clientName);
        }
    }

    private function addFromPstnCustomContext(\extensions $ext)
    {
        $section = 'from-pstn-custom';
        $ext->addSectionNoCustom($section, true);

        $extension = '_.';
        $ext->add($section, $extension, '', new \ext_execif('$[${CALLERID(num):0:1}=0]', 'Set', 'CALLERID(num)=+375${CALLERID(num):1}'));
        $ext->add($section, $extension, '', new \ext_execif('$[${CALLERID(num):0:2}=80]', 'Set', 'CALLERID(num)=+375${CALLERID(num):2}'));
        $ext->add($section, $extension, '', new \ext_execif('$[${CALLERID(num):0:1}=8]', 'Set', 'CALLERID(num)=+375${CALLERID(num):1}'));
        $ext->add($section, $extension, '', new \ext_gosubif('$["${CALLERID(num)}" = "Unknown"]', 'zamena,${EXTEN},1'));
        $ext->add($section, $extension, '', new \ext_set('numdig', '${LEN(${CALLERID(num)})}'));
        $ext->add($section, $extension, '', new \ext_gosubif('$["${numdig}" = "7"]', 'numberadddigit,${EXTEN},1'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(number)', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(all)', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_setcidname('${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('__clientnum', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('__calldirection', 'in'));
        $ext->add($section, $extension, '', new \ext_agi('accountuniqe.sh'));
        $ext->add($section, $extension, '', new \ext_set('CHANNEL(accountcode)', '${bashres}'));
        $ext->add($section, $extension, '', new \ext_set('__varnum', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_dbput('${EXTEN}/${CALLERID(num)}/account', '${CHANNEL(accountcode)}'));

        if ('' !== $this->getClientName()) {
            $ext->add($section, $extension, 'ats2bx24-incall', new \ext_system(
                $this->getBinFileExec('inbound.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${EXTEN}"'
                .' "${CALLERID(num)}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y %H:%M:%S)}"'
                .' "${CHANNEL(accountcode)}"'
                .' &'
            ));
        }

        $ext->add($section, $extension, '', new \ext_dbget('dialnum', 'sootv/${EXTEN}'));
        $ext->add($section, $extension, '', new \ext_noop('dialnum = ${dialnum}'));
        $ext->add($section, $extension, '', new \ext_set('CHANNEL(hangup_handler_push)', 'in-call-end,s,1'));
        $ext->add($section, $extension, '', new \ext_dbput('${EXTEN}/${CALLERID(num)}/queuecount', '0'));
        $ext->add($section, $extension, '', new \ext_set('__startcdrnew', '${STRFTIME(${EPOCH},UTC+0,%Y-%m-%d %H:%M:%S)}'));
        $ext->add($section, $extension, '', new \ext_goto('1', '${EXTEN}', 'new-ext-did'));

        $extension = 'h';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_hangup());
    }

    private function addNewExtDidContext(\extensions $ext)
    {
        $section = 'new-ext-did';
        $ext->addSectionNoCustom($section, true);

        $ext->addInclude($section, 'ext-did');
        $ext->addInclude($section, 'ext-did-post-custom');
        $ext->addInclude($section, 'from-did-direct');
        $ext->addInclude($section, 'ext-did-catchall');

        $extension = 'h';
        $ext->add($section, $extension, '', new \ext_hangup());
    }

    private function addZamenaContext(\extensions $ext)
    {
        $section = 'zamena';
        $ext->addSectionNoCustom($section, true);

        $extension = '_.';
        $ext->add($section, $extension, '', new \ext_set('CALLERID(num)', '+375171234567'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(all)', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_setcidname('${CALLERID(all)}'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addNumberadddigitContext(\extensions $ext)
    {
        $section = 'numberadddigit';
        $ext->addSectionNoCustom($section, true);

        $extension = '_.';
        $ext->add($section, $extension, '', new \ext_set('CALLERID(num)', '+37517${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addMisscallContext(\extensions $ext)
    {
        $section = 'misscall';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());

        if ('' !== $this->getClientName()) {
            $ext->add($section, $extension, 'ats2bx24-misscall', new \ext_system(
                $this->getBinFileExec('miss.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CALLERID(num)}"'
                .' "${FROM_DID}"'
                .' "${DIALEDPEERNAME:4:3}"'
                .' "${ANSWEREDTIME}"'
                .' "${DIALSTATUS}"'
                .' "${MIXMONITOR_FILENAME}"'
                .' "${UNIQUEID}"'
                .' "${CHANNEL(accountcode)}"'
                .' &'
            ));
        }

        $ext->add($section, $extension, '', new \ext_set('ODBC_NEWCDMISS(${startcdrnew},${clientnum},,${FROM_DID},${NODEST},NO ANSWER,${CHANNEL(LINKEDID)},${CHANNEL(accountcode)},${CDR(duration)},${timecall},${CALLFILENAME}.mp3)'));
        $ext->add($section, $extension, '', new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/num)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/account)}'));
        $ext->add($section, $extension, 'dbdel', new \ext_gosub('1', 's', 'misscalldbdel'));
        $ext->add($section, $extension, 'userfieldmod', new \ext_set('CDR(userfield)', 'miss_call'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addMisscalldbdelContext(\extensions $ext)
    {
        $section = 'misscalldbdel';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/num)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/account)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queuecount)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queue)}'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addMacroOutdialContext(\extensions $ext)
    {
        $section = 'macro-outdial';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());

        if ('' !== $this->getClientName()) {
            $ext->add($section, $extension, 'ats2bx24-outcall-answ', new \ext_system(
                $this->getBinFileExec('outound.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${CONNECTEDLINE(num)}"'
                .' "${CALLERID(num)}"'
                .' "${CDR(start)}"'
                .' "${CHANNEL(accountcode)}"'
                .' &'
            ));
        }
    }

    private function addCallanswerContext(\extensions $ext)
    {
        $section = 'callanswer';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop('"${LEN(${CALLERID(num)})}" "${CALLERID(num):0:1}"'));
        $ext->add($section, $extension, '', new \ext_execif('$["${clientnum}" = ""]', 'Return'));
        $ext->add($section, $extension, '', new \ext_dbput('${FROM_DID}/${clientnum}/num', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_dbget('CallStart', '${FROM_DID}/${CALLERID(num)}/num/callstart'));
        $ext->add($section, $extension, '', new \ext_execif('$["${CallStart}" = ""]', 'Set', 'DB(${FROM_DID}/${clientnum}/num/callstart)=${STRFTIME(epoch,,%s)}'));
        $ext->add($section, $extension, '', new \ext_dbget('AC', '${FROM_DID}/${clientnum}/account'));
        $ext->add($section, $extension, 'giveaccountcode', new \ext_execif('$["${AC}"=""]', 'Set', 'AC=${variac}'));

        if ('' !== $this->getClientName()) {
            $ext->add($section, $extension, 'ats2bx24-answer', new \ext_system(
                $this->getBinFileExec('answer.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${FROM_DID}"'
                .' "${CALLERID(num)}"'
                .' "${clientnum}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y %H:%M:%S)}"'
                .' "${AC}"'
                .' &'
            ));
        }

        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addCallansweContext(\extensions $ext)
    {
        $section = 'callanswe';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop('"${LEN(${CALLERID(num)})}" "${CALLERID(num):0:1}"'));
        $ext->add($section, $extension, '', new \ext_dbput('${FROM_DID}/${clientnum}/num', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_dbget('CallStart', '${FROM_DID}/${CALLERID(num)}/num/callstart'));
        $ext->add($section, $extension, '', new \ext_execif('$["${CallStart}" = ""]', 'Set', 'DB(${FROM_DID}/${clientnum}/num/callstart)=${STRFTIME(epoch,,%s)}'));
        $ext->add($section, $extension, '', new \ext_dbget('AC', '${FROM_DID}/${clientnum}/account'));
        $ext->add($section, $extension, 'giveaccountcode', new \ext_execif('$["${AC}"=""]', 'Set', 'AC=${variac}'));

        if ('' !== $this->getClientName()) {
            $ext->add($section, $extension, 'ats2bx24-answer', new \ext_system(
                $this->getBinFileExec('answer.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${FROM_DID}"'
                .' "${CALLERID(num)}"'
                .' "${clientnum}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y %H:%M:%S)}"'
                .' "${AC}"'
                .' &'
            ));
        }

        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addInCallEndContext(\extensions $ext)
    {
        $section = 'in-call-end';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_gosubif('$["${DB(${FROM_DID}/${CALLERID(num)}/num)}" = ""]', 'misscall,s,1'));
        $ext->add($section, $extension, '', new \ext_dbget('innumber', '${FROM_DID}/${CALLERID(num)}/num'));
        $ext->add($section, $extension, '', new \ext_dbget('CallStart', '${FROM_DID}/${CALLERID(num)}/num/callstart'));
        $ext->add($section, $extension, '', new \ext_set('CallStop', '${STRFTIME(epoch,,%s)}'));
        $ext->add($section, $extension, '', new \ext_set('timecall', '${MATH(${CallStop}-${CallStart},int)}'));
        $ext->add($section, $extension, '', new \ext_set('memrealnum', '${CUT(CUT(MEMBERINTERFACE,/,2),@,1)}'));
        $ext->add($section, $extension, '', new \ext_noop('${memrealnum}'));
        $ext->add($section, $extension, '', new \ext_execif('$["${innumber}" != "${memrealnum}" & "${memrealnum}" != ""]', 'Set', '__innumber=${memrealnum}'));
        $ext->add($section, $extension, '', new \ext_execif('$["${innumber}" != ""]', 'Set', 'ODBC_NEWCDRIN(${startcdrnew},${clientnum},${innumber},${FROM_DID},${NODEST},ANSWERED,${CHANNEL(LINKEDID)},${CHANNEL(accountcode)},${CDR(duration)},${timecall},${CALLFILENAME}.mp3)='));
        $ext->add($section, $extension, '', new \ext_execif('$["${innumber}" != ""]', 'Set', 'ODBC_UPDATEMISS(${clientnum},${CHANNEL(accountcode)})='));

        if ('' !== $this->getClientName()) {
            $ext->add($section, $extension, 'send', new \ext_execif(
                '$["${innumber}" != ""]',
                'System',
                $this->getBinFileExec('hangup.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CALLERID(num)}"'
                .' "${FROM_DID}"'
                .' "${DB(${FROM_DID}/${CALLERID(num)}/num)}"'
                .' "${timecall}"'
                .' "ANSWERED"'
                .' "/var/spool/asterisk/monitor/${YEAR}/${MONTH}/${DAY}/${CALLFILENAME}.mp3"'
                .' "${CHANNEL(LINKEDID)}"'
                .' "${DB(${FROM_DID}/${CALLERID(num)}/account)}"'
                .' &'
            ));
        }

        $ext->add($section, $extension, 'dbdel', new \ext_gosubif('$[${innumber}" != ""]', 'in-call-end-dbdel,s,1'));
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addInCallEndDbdelContext(\extensions $ext)
    {
        $section = 'in-call-end-dbdel';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/num/callstart)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/account)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queuecount)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/num)}'));
        $ext->add($section, $extension, '', new \ext_noop('${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queue)}'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addOutnewCdr(\extensions $ext)
    {
        $section = 'outnew-cdr';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop('${CUT(custom,/,2)}'));

        // звонки с сип транков
        $ext->add($section, $extension, '', new \ext_noop('${CUT(custom,/,2)}'));

        // звонки с кастомного транка
        $ext->add($section, $extension, '', new \ext_noop('${CUT(OUT_${DIAL_TRUNK},$,3)}'));

        // звонки с кастомного транка
        $ext->add($section, $extension, '', new \ext_noop('${CDR(dstchannel)}'));

        // звонки с кастомного транка
        $ext->add($section, $extension, '', new \ext_set('pjsipstatus', '${CUT(CDR(dstchannel),/,1)}'));

        $ext->add($section, $extension, '', new \ext_execif('$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" = "" & "${trunkname}" = "" & "${pjsipstatus}" = "PJSIP"]', 'Set', '__trunkname=${CUT(CUT(CDR(dstchannel),/,2),-,1)}'));
        $ext->add($section, $extension, '', new \ext_execif('$["${calldirection}" = "out" & "${CUT(custom,/,2)}" = "" & "${CUT(OUT_${DIAL_TRUNK},$,3)}" = "" & "${pjsipstatus}" != "PJSIP"]', 'Set', '__trunkname=${CUT(pre_num,/,2)}'));        $ext->add($section, $extension, '', new \ext_execif('$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" = "" & "${trunkname}" = "" & "${pjsipstatus}" != "PJSIP"]', 'Set', '__trunkname=${CUT(custom,/,2)}'));
        $ext->add($section, $extension, '', new \ext_execif('$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" != "" & "${trunkname}" = "" & "${pjsipstatus}" != "PJSIP"]', 'Set', '__trunkname=${CUT(CUT(post_num,@,2),/,1)}'));
        $ext->add($section, $extension, '', new \ext_noop('${trunkname}'));
        $ext->add($section, $extension, '', new \ext_execif('$["${trunkname}" != ""]', 'Set', 'ODBC_NEWCDROUT(${startcdrnew},${REALCALLERIDNUM},${numbclient},${trunkname},${CDR(disposition)},${CHANNEL(LINKEDID)},${CHANNEL(accountcode)},${CDR(duration)},${CDR(billsec)},${CALLFILENAME}.mp3)=111'));
        $ext->add($section, $extension, '', new \ext_execif('$["${trunkname}" != "" & "${CDR(disposition)}" = "ANSWER"]', 'Set', 'ODBC_UPDATEMISS(${numbclient},${CHANNEL(accountcode)})=111'));
        $ext->add($section, $extension, '', new \ext_execif('$["${trunkname}" != "" & "${CDR(disposition)}" = "ANSWERED"]', 'Set', 'ODBC_UPDATEMISS(${numbclient},${CHANNEL(accountcode)})=111'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function modifyMacroHangupcallContext(\extensions $ext)
    {
        $section = 'macro-hangupcall';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }

        $extension = 's';

        // region Prepend commands before "skipagi" tag
        $beforeTag = 'skipagi';
        $ext->splice($section, $extension, $beforeTag, new \ext_dbget('realnum', '${FROM_DID}/${CALLERID(num)}/num'));
        $ext->splice($section, $extension, $beforeTag, new \ext_noop('from_did=${FROM_DID} calldirection=${calldirection} mixmonfile=${MIXMONITOR_FILENAME}'));

        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, $beforeTag, new \ext_execif(
                '$["${MIXMONITOR_FILENAME}" != "" & "${calldirection}" = "out" & "${FROM_DID}" = ""]',
                'System',
                $this->getBinFileExec('hangup2.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CALLERID(num)}"'
                .' "${FROM_DID}"'
                .' "${realnum}"'
                .' "${CDR(billsec)}"'
                .' "${CDR(disposition)}"'
                .' "/var/spool/asterisk/monitor/${YEAR}/${MONTH}/${DAY}/${CALLFILENAME}.mp3"'
                .' "${UNIQUEID}"'
                .' "${CHANNEL(accountcode)}"'
                .' &'
            ), 'ats2bx24-hangup');
        }

        $ext->splice($section, $extension, $beforeTag, new \ext_execif('$["${MIXMONITOR_FILENAME}" != "" & "${calldirection}" = "in"]', 'Noop', '${DB(${FROM_DID}/${CALLERID(num)}/account)}'));
        $ext->splice($section, $extension, $beforeTag, new \ext_execif('$["${MIXMONITOR_FILENAME}" != "" & "${calldirection}" = "in"]', 'Noop', '${DB(${FROM_DID}/${CALLERID(num)}/num)}'));
        $ext->splice($section, $extension, $beforeTag, new \ext_gosubif('$["${calldirection}" = "out"]', 'outnew-cdr,s,1'));
        // endregion Prepend commands before "skipagi" tag
    }

    private function modifyMacroDialoutTrunk(\extensions $ext)
    {
        $section = 'macro-dialout-trunk';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('В исходящих маршрутах отсутствуют транки: Required "%s" section not found', $section));
        }

        $extension = 's';

        // region Appending lines after first command
        $afterPriority = 1;
        $ext->splice($section, $extension, $afterPriority++, new \ext_agi('accountuniqeout.sh'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_set('__calldirection', 'out'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_set('__accountcode', '${CHANNEL(accountcode)}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_set('__startcdrnew', '${STRFTIME(${EPOCH},UTC+0,%Y-%m-%d %H:%M:%S)}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode)}" = ""]', 'Set', 'CHANNEL(accountcode)=${accoutout}'));
        // endregion Appending lines after first command

        // region Replace "Set(DIAL_NUMBER=...)" command by custom "Set(DIAL_NUMBER=...)" command
        $insteadOfPriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_setvar) {
                if ('DIAL_NUMBER' === $command['cmd']->var) {
                    $insteadOfPriority = $priority + 1;
                    break;
                }
            }
        }
        if (null === $insteadOfPriority) {
            throw new \UnexpectedValueException(sprintf('Unable to find %s@%s extension\'s "Set(DIAL_NUMBER=...)" command', $extension, $section));
        }
        $ext->replace($section, $extension, $insteadOfPriority, new \ext_set('DIAL_NUMBER', '${FILTER(\+0123456789,${ARG2})}'));
        // endregion Replace "Set(DIAL_NUMBER=...)" by custom "Set(DIAL_NUMBER=...)"

        // region Append custom commands after "GotoIf(...?customtrunk)" command
        $afterPriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_gotoif) {
                if ('customtrunk' === $command['cmd']->true_priority) {
                    $afterPriority = $priority + 1;
                    break;
                }
            }
        }
        if (null === $afterPriority) {
            throw new \UnexpectedValueException(sprintf('Unable to find %s@%s extension\'s "GotoIf(...?customtrunk)"', $extension, $section));
        }

        $ext->splice($section, $extension, $afterPriority++, new \ext_set('numbclient', '${CONNECTEDLINE(num)}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${numbclient:0:3}" = "375"]', 'Set', 'numbclient=+375${numbclient:3}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${numbclient:0:2}" = "80"]', 'Set', 'numbclient=+375${numbclient:2}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${numbclient:0:2}" = "00"]', 'Set', 'numbclient=+375${numbclient:2}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${numbclient:0:1}" = "7"]', 'Set', 'numbclient=+7${numbclient:1}'));

        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, $afterPriority++, new \ext_system(
                $this->getBinFileExec('outound2.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${CONNECTEDLINE(num)}"'
                .' "${CALLERID(num)}"'
                .' "${CDR(start)}"'
                .' "${CHANNEL(accountcode)}"'
                .' "${zamena}"'
                .' &'
            ));
        }
        // endregion Append lines after "GotoIf(...?customtrunk)"

        // region Prepend "System(.../outound2.sh ...)" command as first command of "skipoutnum" tag
        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, 'skipoutnum', new \ext_system(
                $this->getBinFileExec('outound2.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${CONNECTEDLINE(num)}"'
                .' "${CALLERID(num)}"'
                .' "${CDR(start)}"'
                .' "${CHANNEL(accountcode)}"'
                .' "${zamena}"'
                .' &'
            ), 'skipoutnum');
        }
        // endregion Prepend "System(.../outound2.sh ...)" command as first command of "skipoutnum" tag

        $extension = 's-BUSY';

        // region Replace "Busy(...)" command by "Macro(...)" command
        $insteadOfPriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_busy) {
                $insteadOfPriority = $priority + 1;
                break;
            }
        }
        if (null === $insteadOfPriority) {
            throw new \UnexpectedValueException(sprintf('Unable to find %s@%s extension\'s "Busy(...)"', $extension, $section));
        }
        $ext->replace($section, $extension, $insteadOfPriority, new \ext_macro('hangupcall'));
        // endregion Replace "Busy(...)" command by "Macro(...)" command
    }

    private function modifyMacroDialOneContext(\extensions $ext)
    {
        $section = 'macro-dial-one';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }

        $extension = 's';

        // region Prepend commands as first command of "dial" tag
        $inFrontOfTag = 'dial';

        $afterPriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($inFrontOfTag === $command['tag']) {
                $afterPriority = $priority; // First found "n(dial)" priority - 1
                break;
            }
        }
        if (null === $afterPriority) {
            throw new \UnexpectedValueException(sprintf('Unable to find %s@%s extension\'s "dial" tag', $extension, $section));
        }

        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('QUEUE_VARIABLES(NODEST)'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" = ""]', 'Set', 'DB(${FROM_DID}/${CALLERID(num)}/queue)=${NODEST}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" != "${NODEST}"]', 'Set', 'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=0'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" != "${NODEST}"]', 'Set', 'DB(${FROM_DID}/${CALLERID(num)}/queue)=${NODEST}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_set('ringallstrat', 'NO'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_set('strat', '${SHELL(asterisk -rx \'queue show ${NODEST}\' | grep ringall| awk \'{print$1}\' | tr -d \'\r\n\')}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${strat}" = "${NODEST}"]', 'Set', 'ringallstrat=YES'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('"${strat}" = "${NODEST}"'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${ringallstrat}" = "YES"]', 'Set', 'dstext=${FILTER(\,0123456789,${QUEUE_MEMBER_LIST(${NODEST})})}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${ringallstrat}" != "YES"]', 'Set', 'memcount=${QUEUE_MEMBER_COUNT(${NODEST})}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${FROMQ} ${AGENT_DEST} FROMQ=${FROMQ} strategy=${ringallstart} Nodest=${NODEST} Members=${QUEUE_MEMBER_LIST(${NODEST})} Members2=${FILTER(\,0123456789,${QUEUE_MEMBER_LIST(${NODEST})})} membercount=${QUEUE_MEMBER_COUNT(${NODEST})}'), $inFrontOfTag);

        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, $afterPriority++, new \ext_execif(
                '$["${clientnum}" != "" & "${dstext}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount):-0}" < 1 ]',
                'System',
                $this->getBinFileExec('queue.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${clientnum}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}"'
                .' "${CHANNEL(accountcode)}"'
                .' "${dstext}"'
                .' &'
            ), $inFrontOfTag);
        }

        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${clientnum}" != "" & "${dstext}" != "" ]', 'Set', 'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=${MATH(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}+1,int)}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'), $inFrontOfTag);

        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, $afterPriority++, new \ext_execif(
                '$["${clientnum}" != "" & "${dstext}" = "" & ${DB(${FROM_DID}/${CALLERID(num)}/queuecount)} < ${memcount}]',
                'System',
                $this->getBinFileExec('queue.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${clientnum}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}"'
                .' "${CHANNEL(accountcode)}"'
                .' "${ARG3}"'
                .' &'
            ), $inFrontOfTag);
        }

        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${clientnum}" != "" & "${dstext}" = "" & ${DB(${FROM_DID}/${CALLERID(num)}/queuecount)} < ${memcount}]', 'Set', 'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=${MATH(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}+1,int)}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'), $inFrontOfTag);

        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, $afterPriority++, new \ext_execif(
                '$["${clientnum}" != "" & "${dstext}" = "" & "${ringallstrat}" != "YES" & "${memcount}" = "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount):-0}" < 1]',
                'System',
                $this->getBinFileExec('queue.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${clientnum}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}"'
                .' "${CHANNEL(accountcode)}"'
                .' "${ARG3}"'
                .' &'
            ), $inFrontOfTag);
        }

        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${clientnum}" != "" & "${dstext}" = "" & "${ringallstrat}" != "YES" & "${memcount}" = ""]', 'Set', 'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=${MATH(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}+1,int)}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'), $inFrontOfTag);
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${LEN(${CALLERID(num)})}'), $inFrontOfTag);
        // endregion Prepend commands as first command of "dial" tag
    }

    private function modifyMacroDialContext(\extensions $ext)
    {
        $section = 'macro-dial';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }

        $extension = 's';

        // region Append command after "Noop(Working with ...)" command
        $afterPriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_noop) {
                if (0 === strpos($command['cmd']->data, 'Working with ')) {
                    $afterPriority = $priority + 1; // First found "Noop(Working with ...)" command priority
                    break;
                }
            }
        }
        if (null === $afterPriority) {
            throw new \UnexpectedValueException(sprintf('Unable to find %s@%s extension\'s "Noop(Working with ...)" command', $extension, $section));
        }

        if ('' !== $this->getClientName()) {
            $ext->splice($section, $extension, $afterPriority++, new \ext_execif(
                '$["${clientnum}" != "" & "${dstext}" = ""]',
                'System',
                $this->getBinFileExec('queue.sh', array('klient' => '${ZRUCHNA_CLIENT_NAME}'))
                .' "${CHANNEL(LINKEDID)}"'
                .' "${clientnum}"'
                .' "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}"'
                .' "${CHANNEL(accountcode)}"'
                .' "${ARG3}"'
                .' &'
            ));
        }
        // endregion Append command after "Noop(Working with ...)" command
    }

    /**
     * @param string $binFileName
     * @param array<string, scalar> $environments
     * @return string
     */
    private function getBinFileExec($binFileName, array $environments = array())
    {
        $prefix = '';
        foreach ($environments as $name => $value) {
            $prefix .= sprintf('%s="%s" ', $name, strtr($value, array('"' => '\\"')));
        }

        return $prefix.'/bin/bash '.$this->escapeExtData(escapeshellarg(__DIR__.'/bin/'.$binFileName));
    }

    /**
     * @return string
     */
    protected function getClientName()
    {
        return $this->getConfig('client_name');
    }

    private function escapeExtData($value)
    {
        return strtr($value, array(
            ';' => '\;',
            ')' => '\)',
        ));
    }

    public function doConfigPageInit($page)
    {
    }

    public function getActionBar($request)
    {
        $buttons = array();
        switch ($_GET['display']) {
            case 'zruchnaio':
                $buttons = array(
                    'delete' => array(
                        'name' => 'delete',
                        'id' => 'delete',
                        'value' => _('Delete')
                    ),
                    'reset' => array(
                        'name' => 'reset',
                        'id' => 'reset',
                        'value' => _('Reset')
                    ),
                    'submit' => array(
                        'name' => 'submit',
                        'id' => 'submit',
                        'value' => _('Submit')
                    )
                );
                if (empty($_GET['extdisplay'])) {
                    unset($buttons['delete']);
                }
                break;
        }
        return $buttons;
    }

    public function showPage()
    {
        $view = empty($_REQUEST['view']) ? 'main' : $_REQUEST['view'];

        switch ($view) {
            case 'setup':
                switch ($_SERVER['REQUEST_METHOD']) {
                    case 'GET':
                        return load_view(__DIR__ . '/views/setup.php', [
                            'freepbxBaseUrl' => $this->getConfig('freepbx_base_uri'),
                            'recordsExtension' => $this->getConfig('records_extension'),
                            'freepbxCallbackExtension' => $this->getConfig('freepbx_callback_extension'),
                            'clientName' => $this->getClientName(),
                            'error' => isset($_GET['error']) ? $_GET['error'] : '',
                        ]);
                    case 'POST':
                        $this->setConfig('records_extension', $_REQUEST['records_extension']);

                        $_SERVER['REQUEST_URI'] = preg_replace(
                            ['~error(?:=[^&#]*)?~', '~&{2,}~'],
                            '&',
                            $_SERVER['REQUEST_URI']
                        );

                        if (!empty($_REQUEST['bx24_application_secret'])) {
                            if ($this->touchIntegrationConfigs($_REQUEST['bx24_application_secret'], $_REQUEST['freepbx_base_uri'], $_REQUEST['freepbx_callback_extension'])) {
                                $this->setConfig('freepbx_base_uri', $_REQUEST['freepbx_base_uri']);
                                $this->setConfig('freepbx_callback_extension', $_REQUEST['freepbx_callback_extension']);
                            } else {
                                $_SERVER['REQUEST_URI'] .= false !== strpos($_SERVER['REQUEST_URI'], '?') ? '&' : '?';
                                $_SERVER['REQUEST_URI'] .= 'error=failed-public-url';
                            }
                        }

                        \needreload();

                        header('Location: '.$_SERVER['REQUEST_URI'], true, 302);
                        return '';
                }
            default:
                $vars = array('helloworld' => _("Hello World"));
                return load_view(__DIR__ . '/views/main.php', $vars);
        }
    }

    /**
     * @param string $bx24Secret
     * @param string $freepbxBaseUri
     * @param string $freepbxCallbackExtension
     * @throws \RuntimeException
     * @return bool
     *
     * @api
     */
    private function touchIntegrationConfigs($bx24Secret, $freepbxBaseUri, $freepbxCallbackExtension)
    {
        $ch = curl_init('https://asterisk-connect.zruchna.io/freepbx/api/v1/registration');
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
            ],
            CURLOPT_POSTFIELDS => http_build_query([
                'registration_info' => [
                    'freepbx_module_secret' => $this->getSecretValue(),
                    'bitrix24_app_secret' => $bx24Secret,
                    'freepbx_base_uri' => $freepbxBaseUri,
                    'freepbx_callback_extension' => $freepbxCallbackExtension,
                ],
            ], '', '&', PHP_QUERY_RFC3986),
        ]);
        $responseBody = curl_exec($ch);
        $statusCode = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        $errorCode = curl_errno($ch);
        $errorMessage = curl_error($ch);
        curl_close($ch);

        if (CURLE_OK !== $errorCode) {
            throw new \RuntimeException('Unable to touch integration config: cURL ['.$errorCode.'] '.$errorMessage);
        }

        $responsePayload = json_decode($responseBody, true);

        if ($statusCode < 200 || $statusCode >= 300) {
            throw new \RuntimeException('Unable to touch integration config: ['.$statusCode.']: '.$responsePayload['message']);
        }

        if (isset($responsePayload['clientName'])
            && is_string($responsePayload['clientName'])
        ) {
            $this->setConfig('client_name', $responsePayload['clientName']);
        }

        return $responsePayload['status'];
    }

    /**
     * @param string $req
     * @param array $setting
     * @return bool
     *
     * @noinspection PhpUnused
     */
    public function ajaxRequest($req, &$setting)
    {
        switch ($req) {
            case 'ping':
            case 'is_signed':
                $this->allowAnonymous($setting);

                return true;
            case 'originate':
                if ('POST' !== $_SERVER['REQUEST_METHOD']) {
                    return false;
                }

                $this->allowAnonymous($setting);

                if (!isset($_GET['secret'])) {
                    header('X-Reject-Reason: Miss required parameter "secret"', false);

                    return false;
                }
                if ($this->getSecretValue() !== $_GET['secret']) {
                    header('X-Reject-Reason: Invalid value of "secret" parameter', false);

                    // To prevent bruteforce.
                    sleep(2);

                    return false;
                }

                return true;
            case 'record':
                if ('GET' !== $_SERVER['REQUEST_METHOD']) {
                    header('X-Reject-Reason: Non-supported request method', false);

                    return false;
                }

                $this->allowAnonymous($setting);

                if (!isset($_GET['secret'])) {
                    header('X-Reject-Reason: Miss required parameter "secret"', false);

                    return false;
                }
                if ($this->getSecretValue() !== $_GET['secret']) {
                    header('X-Reject-Reason: Invalid value of "secret" parameter', false);

                    // To prevent bruteforce.
                    sleep(2);

                    return false;
                }
                if (!isset($_GET['call_id'])) {
                    header('X-Reject-Reason: Miss required parameter "call_id"', false);

                    return false;
                }
                if (!\is_string($_GET['call_id'])) {
                    header(sprintf('X-Reject-Reason: Expected to "call_id" parameter be string, %s provided', gettype($_GET['call_id'])), false);

                    return false;
                }

                return true;
            default:
                return false;
        }
    }

    public function ajaxHandler()
    {
        switch ($_REQUEST['command']) {
            case 'ping':
                return 'pong';
            case 'is_signed':
                if (!isset($_GET['secret'])) {
                    header('X-Reject-Reason: Miss required parameter "secret"', false);

                    return 'no';
                }

                // To prevent bruteforce.
                sleep(2);

                if ($this->getSecretValue() !== $_GET['secret']) {
                    header('X-Reject-Reason: Invalid value of "secret" parameter', false);

                    return 'no';
                }

                return 'yes';
            case 'originate':
                $extension = isset($_POST['extension']) ? $_POST['extension'] : null;
                $phone = isset($_POST['phone']) ? $_POST['phone'] : null;
                $callId = isset($_POST['call_id']) ? $_POST['call_id'] : null;
                $lineNumber = isset($_POST['line_number']) ? $_POST['line_number'] : null;

                if (empty($extension)) {
                    return false;
                }
                if (empty($phone)) {
                    return false;
                }
                if (empty($callId)) {
                    return false;
                }

                $response = $this->astman->Originate([
                    'Channel' => sprintf('Local/%s@from-internal', $extension),
                    'CallerID' => sprintf('%s', $extension),
                    'Timeout' => '20000',
                    'Account' => sprintf('%s', $callId),
                    'Context' => 'from-internal',
                    'Exten' => sprintf('%s', $phone),
                    'Priority' => '1',
                    'Variable' => [
                        'variac' => sprintf('%s', $callId),
                        'line' => sprintf('%s', $lineNumber),
                        'calldirection' => 'out',
                    ],
                ]);

                $response = array_change_key_case($response, \CASE_LOWER);
                if (strtolower($response['response']) === 'error') {
                    return sprintf('Error: %s', $response['message']);
                }

                return 'Done';
            case 'record':
                $callId = isset($_GET['call_id']) ? $_GET['call_id'] : null;
                if (!preg_match('~^[1-9]\d*[._]\d+$~', $callId)) {
                    header('Content-Type: application/json', true, 400);
                    echo '{"status":false,"message":"Invalid call ID value provided"}';
                    exit;
                }

                $cdr = \FreePBX::Cdr();
                \assert($cdr instanceof Cdr);

                if ([] === $cdrRecord = $cdr->getRecordByID($callId)) {
                    header('Content-Type: application/json', true, 404);
                    echo '{"status":false,"message":"Unable to find CDR record"}';
                    exit;
                }
                if (empty($cdrRecord['uniqueid'])) {
                    header('Content-Type: application/json', true, 404);
                    echo '{"status":false,"message":"Unable to find CDR record"}';
                    exit;
                }

                $recordFile = $cdrRecord['recordingfile'];
                if (empty($recordFile)) {
                    header('Content-Type: application/json', true, 404);
                    echo '{"status":false,"message":"CDR record does not contain \"recordingfile\" value"}';
                    exit;
                }

                $callId = strtr($callId, '_', '.');

                $recordFileMime = mime_content_type($recordFile) ?: 'application/octet-stream';
                $recordFileSize = filesize($recordFile);
                $recordFileSafeFilename = sprintf('"record %s.%s"', $callId, pathinfo($recordFile, PATHINFO_EXTENSION));
                $recordFileRfc5987Filename = sprintf('UTF-8\'\'%s', rawurlencode(pathinfo($recordFile, PATHINFO_BASENAME)));

                session_write_close();

                while (ob_get_level()) {
                    @ob_end_clean();
                }

                header_remove();
                header(sprintf('Content-Type: %s', $recordFileMime));
                header(sprintf('Content-Length: %d', $recordFileSize));
                header(sprintf('Content-Disposition: attachment; filename=%s; filename*=%s', $recordFileSafeFilename, $recordFileRfc5987Filename));
                header('Cache-Control: private, max-age=2592000'); // =60×60×24×30 (1 month)
                header(sprintf('Expires: %s', \gmdate('D, d M Y H:i:s \G\M\T', strtotime('now +1 month'))));
                header('Connection: close');

                readfile($recordFile);
                exit;
            default:
                return false;
        }
    }

    public function getRightNav($request)
    {
        return <<<'HTML'
<div class="bootstrap-table">
  <div class="fixed-table-toolbar">
    <div class="bs-bars">
      <div class="toolbar-cbbnav">
        <a href="?display=zruchnaio&view=setup" class="btn btn-default">Setup</a>
      </div>
    </div>
  </div>
</div>
HTML;
    }

    /**
     * @return void
     */
    private function allowAnonymous(array &$setting)
    {
        $setting['allowremote'] = true;
        $setting['authenticate'] = false;
    }

    /**
     * @return string|null
     */
    private function getSecretValue()
    {
        try {
            return $this->getConfig('freepbx_module_secret') ?: null;
        } catch (\Exception $e) {
            throw new \LogicException('Impossible error: '.$e->getMessage(), 0, $e);
        }
    }
}
