<?php

namespace FreePBX\Modules;

use FreePBX\Freepbx_conf;

class Zruchnaio extends \FreePBX_Helpers implements \BMO
{
    /** @var \FreePBX $FreePBX */
    protected $FreePBX;

    /** @var \DB */
    protected $db;

    /** @var Freepbx_conf */
    private $freePbxConf;

    /**
     * @var \AGI_AsteriskManager
     */
    private $astman;

    /**
     * @param \FreePBX $freepbx
     * @throws \Exception
     * @throws \BadMethodCallException
     */
    public function __construct($freepbx = null)
    {
        parent::__construct($freepbx);

        if (null === $freepbx) {
            throw new \BadMethodCallException('Not given a FreePBX Object');
        }

        $this->FreePBX = $freepbx;
        $this->db = $freepbx->Database;
        $this->freePbxConf = Freepbx_conf::create();
        $this->astman = $freepbx->astman;
    }

    /**
     * Set Priority for Dialplan Hooking
     * Core sits at a priority of 600
     * @method myDialplanHooks
     *
     * @return string        Priority
     */
    public static function myDialplanHooks()
    {
        return 900;
    }

    public function install()
    {
        // global $amp_conf;

        $this->setMultiConfig([
            'freepbx_module_secret' => $this->createSecret(),
            'freepbx_base_uri' => '',
            'records_extension' => 'wav',
            'freepbx_callback_extension' => '',
            'client_name' => '',
        ]);

        // $htdocs_dest = $amp_conf['AMPWEBROOT'];
        // $bin_dest    = isset($amp_conf['AMPBIN']) ? $amp_conf['AMPBIN'] : '/var/lib/asterisk/bin';
        // $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $this->addAgiBinFile('accountuniqe');
        $this->addAgiBinFile('accountuniqeout');

        $this->freePbxConf->set_conf_values(array(
            'DIAL_OPTIONS' => 'HhTtr',
            'TRUNK_OPTIONS' => 'TB(outdialanswer-bx24,s,1)',
        ));
        $this->freePbxConf->commit_conf_settings();
    }

    /**
     * @return string
     */
    private function createSecret()
    {
        return base64_encode(openssl_random_pseudo_bytes(16));
    }

    public function uninstall()
    {
        $this->removeAgiBinFile('accountuniqe');
        $this->removeAgiBinFile('accountuniqeout');
    }

    private function addAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $srcFilePath = __DIR__.'/agi/'.$filename.'.sh';
        $dstFilePath = $agibin_dest.'/'.$filename.'.sh';

        if (file_exists($dstFilePath)) {
            return;
        }

        symlink($srcFilePath, $dstFilePath);

        if (file_exists($dstFilePath) && !is_executable($dstFilePath)) {
            if (false !== $dstFileStat = stat($dstFilePath)) {
                $dstFileMode = $dstFileStat['mode'] | 0111;
                chmod($dstFilePath, $dstFileMode);
            }
        }
    }

    private function removeAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $dstFilePath = $agibin_dest.'/'.$filename.'.sh';

        if (file_exists($dstFilePath)) {
            unlink($dstFilePath);
        }
    }

    public function backup()
    {
    }

    public function restore($backup)
    {
    }

    /**
     * Hook into Dialplan (extensions_additional.conf)
     * @method doDialplanHook
     *
     * @param \extensions $ext The Extensions Class https://wiki.freepbx.org/pages/viewpage.action?pageId=98701336
     * @param string $engine Always Asterisk, Legacy
     * @param string $priority Priority
     */
    public function doDialplanHook($ext, $engine, $priority)
    {
        $this->registerGlobalVariables($ext);

        $this->addFromPstnCustomContext($ext);
        $this->addNewExtDidContext($ext);
        $this->addZamenaContext($ext);
        $this->addNumberadddigitContext($ext);
        $this->addMisscallContext($ext);
        $this->addMisscalldbdelContext($ext);
        $this->addMacroOutdialContext($ext);
        $this->addInCallEndContext($ext);
        $this->addInCallEndDbdelContext($ext);
        $this->addOuthangupContext($ext);
        $this->addInCallAnswerBx24Context($ext);

        $this->modifyMacroHangupcallContext($ext);
        $this->modifyMacroDialoutTrunk($ext);
        $this->modifyMacroDialOneContext($ext);
        $this->modifyMacroDialContext($ext);
        $this->modifyMacroAutoBlkvmContext($ext);
    }

    private function registerGlobalVariables(\extensions $ext)
    {
        if ('' !== $clientName = $this->getClientName()) {
            $ext->addGlobal('ZRUCHNA_CLIENT_NAME', $clientName);
        }
        if ('' !== $format = $this->getRecordFormat()) {
            $ext->addGlobal('ZRUCHNA_FORMAT_RECORD', $format);
        }
    }

    private function addFromPstnCustomContext(\extensions $ext)
    {
        $section = 'from-pstn-custom';
        $ext->addSectionNoCustom($section, true);
    
        $extension = '_.';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — нормализация номера
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${CALLERID(num):0:1}" = "0"]',
            'Set',
            'CALLERID(num)=+375${CALLERID(num):1}'
        ));
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${CALLERID(num):0:2}" = "80"]',
            'Set',
            'CALLERID(num)=+375${CALLERID(num):2}'
        ));
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${CALLERID(num):0:1}" = "8"]',
            'Set',
            'CALLERID(num)=+375${CALLERID(num):1}'
        ));
    
        // 3 — замена Unknown → zamena-bx24
        $ext->add($section, $extension, '', new \ext_gosubif(
            '$["${CALLERID(num)}" = "Unknown"]',
            'zamena-bx24,${EXTEN},1'
        ));
        $ext->add($section, $extension, '', new \ext_gosubif(
            '$["${CALLERID(num)}" = "UNKNOWN"]',
            'zamena-bx24,${EXTEN},1'
        ));
        $ext->add($section, $extension, '', new \ext_gosubif(
            '$["${CALLERID(num)}" = "anonymous"]',
            'zamena-bx24,${EXTEN},1'
        ));
    
        // 4 — добавление цифры
        $ext->add($section, $extension, '', new \ext_set('numdig', '${LEN(${CALLERID(num)})}'));
        $ext->add($section, $extension, '', new \ext_gosubif(
            '$["${numdig}" = "7"]',
            'numberadddigit-bx24,${EXTEN},1'
        ));
    
        // 5 — установка CID
        $ext->add($section, $extension, '', new \ext_set('CALLERID(number)', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(all)', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(name)', '${CALLERID(num)}'));
    
        // 6 — переменные
        $ext->add($section, $extension, '', new \ext_set('__clientnum', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('__calldirection', 'in'));
    
        // 7 — AGI
        $ext->add($section, $extension, '', new \ext_agi('accountuniqe.sh'));
        $ext->add($section, $extension, '', new \ext_set('CHANNEL(accountcode)', '${bashres}'));
    
        // 8 — запись в базу
        $ext->add($section, $extension, '', new \ext_set(
            'DB(${EXTEN}/${clientnum}/account)',
            '${CHANNEL(accountcode)}'
        ));
    
        // 9 — hangup handler
        $ext->add($section, $extension, '', new \ext_set(
            'CHANNEL(hangup_handler_push)',
            'in-call-end-bx24,s,1'
        ));
    
        // 10 — System(inbound.sh)
        $ext->add($section, $extension, 'ats2bx24-incall', new \ext_execif(
            '$["${DB(zruchna/outcrm/${CALLERID(num)})}" = "" & "${ZRUCHNA_CLIENT_NAME}" != ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/inbound.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${EXTEN}" "${CALLERID(num)}" '
            .'"${STRFTIME(${EPOCH},,%d-%m-%Y %H:%M:%S)}" "${CHANNEL(accountcode)}" &'
        ));
    
        // 11 — queuecount
        $ext->add($section, $extension, '', new \ext_set(
            'DB(${EXTEN}/${clientnum}/queuecount)',
            '0'
        ));
    
        // 12 — время
        $ext->add($section, $extension, '', new \ext_set(
            '__startcdrnew',
            '${STRFTIME(${EPOCH},UTC+0,%Y-%m-%d %H:%M:%S)}'
        ));
        $ext->add($section, $extension, '', new \ext_set(
            '__CallinStart',
            '${STRFTIME(epoch,,%s)}'
        ));
    
        // 13 — переход
        $ext->add($section, $extension, '', new \ext_goto('1', '${EXTEN}', 'new-ext-did-bx24'));
    
        // h
        $ext->add($section, 'h', '1', new \ext_noop());
        $ext->add($section, 'h', '', new \ext_hangup());
    }

    private function addNewExtDidContext(\extensions $ext)
    {
        $section = 'new-ext-did-bx24';
        $ext->addSectionNoCustom($section, true);
    
        // include => ext-did
        $ext->addInclude($section, 'ext-did');
    
        // include => ext-did-post-custom
        $ext->addInclude($section, 'ext-did-post-custom');
    
        // include => from-did-direct
        $ext->addInclude($section, 'from-did-direct');
    
        // include => ext-did-catchall
        $ext->addInclude($section, 'ext-did-catchall');

        $extension = 'h';
        $ext->add($section, $extension, '', new \ext_hangup());
    }

   private function addZamenaContext(\extensions $ext)
    {
        $section = 'zamena-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = '_.';
    
        // 1 — Set(CALLERID(num)=+999999999999)
        $ext->add($section, $extension, '1', new \ext_set(
            'CALLERID(num)',
            '+999999999999'
        ));
    
        // 2 — Set(CALLERID(all)=${CALLERID(num)})
        $ext->add($section, $extension, '', new \ext_set(
            'CALLERID(all)',
            '${CALLERID(num)}'
        ));
    
        // 3 — Set(CALLERID(name)=${CALLERID(all)})
        $ext->add($section, $extension, '', new \ext_set(
            'CALLERID(name)',
            '${CALLERID(all)}'
        ));
    
        // 4 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addNumberadddigitContext(\extensions $ext)
    {
        $section = 'numberadddigit-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = '_.';
    
        // 1 — добавляем код +37517
        $ext->add($section, $extension, '1', new \ext_set(
            'CALLERID(num)',
            '+37517${CALLERID(num)}'
        ));
    
        // 2 — возврат
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addMisscallContext(\extensions $ext)
    {
        $section = 'misscall-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — CallStop
        $ext->add($section, $extension, '', new \ext_set(
            'CallStop',
            '${STRFTIME(epoch,,%s)}'
        ));
    
        // 3 — timecalldur
        $ext->add($section, $extension, '', new \ext_set(
            'timecalldur',
            '${MATH(${CallStop}-${CallinStart},int)}'
        ));
    
        // 4 — если 0 → 1
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${timecalldur}" = "0"]',
            'Set',
            'timecalldur=1'
        ));
    
        // 5 — System(miss.sh)
        $ext->add($section, $extension, 'ats2bx24-misscall', new \ext_execif(
            '$["${ZRUCHNA_CLIENT_NAME}" != ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/miss.sh\' '
            .'"${CALLERID(num)}" "${FROM_DID}" "${DB(${FROM_DID}/${CALLERID(num)}/dstnum)}" '
            .'"${ANSWEREDTIME}" "NO ANSWER" "NoFile" "${UNIQUEID}" '
            .'"${CHANNEL(accountcode)}" "${timecalldur}" "${DB(${FROM_DID}/${CALLERID(num)}/dstext)}" &'
        ));
    
        // 6 — переход в misscall-dbdel-bx24
        $ext->add($section, $extension, '', new \ext_gosub(
            'misscall-dbdel-bx24,s,1'
        ));
    
        // 7 — userfield
        $ext->add($section, $extension, '', new \ext_set(
            'CDR(userfield)',
            'miss_call'
        ));
    
        // 8 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addMisscalldbdelContext(\extensions $ext)
    {
        $section = 'misscall-dbdel-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — DB_DELETE(num)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/num)}'
        ));
    
        // 3 — DB_DELETE(account)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/account)}'
        ));
    
        // 4 — DB_DELETE(queuecount)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queuecount)}'
        ));
    
        // 5 — DB_DELETE(queue)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queue)}'
        ));
    
        // 6 — DB_DELETE(dstnum)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/dstnum)}'
        ));
        
        // 7 — DB_DELETE(dstext)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/dstext)}'
        ));

        // 8 — DB_DELETE(outtotransfer)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${MEMBERNAME}/outtotransfer)}'
        ));
    
        // 9 — DB_DELETE(outtotransfermiss)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${MEMBERNAME}/outtotransfermiss)}'
        ));
    
        // 10 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addMacroOutdialContext(\extensions $ext)
    {
        $section = 'outdialanswer-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — NoOp(${CALLERIDNUMINTERNAL})
        $ext->add($section, $extension, '', new \ext_noop('${CALLERIDNUMINTERNAL}'));
    
        // 3 — ExecIf: запись startcdranswer
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${CHANNEL(accountcode):0:3}" != "ATS"]',
            'Set',
            'DB(out/${CALLERIDNUMINTERNAL}/${numbclient}/startcdranswer)=${STRFTIME(epoch,,%s)}'
        ));
    
        // 4 — System(outanswer.sh)
        $ext->add($section, $extension, 'ats2bx24-outcall-answ', new \ext_execif(
            '$["${ZRUCHNA_CLIENT_NAME}" != ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/outanswer.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${CONNECTEDLINE(num)}" "${CALLERID(num)}" '
            .'"${CDR(start)}" "${CHANNEL(accountcode)}" &'
        ));
    
        // 5 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }    

    private function addInCallEndContext(\extensions $ext)
    {
        $section = 'in-call-end-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — misscall условие
        $ext->add($section, $extension, '', new \ext_gosubif(
            '$["${DB(${FROM_DID}/${CALLERID(num)}/num)}" = "" & "${todial}" = "" & "${DB(${CUT(CUT(MEMBERINTERFACE,/,2),@,1)}/outtotransfer)}" = ""]',
            'misscall-bx24,s,1'
        ));
    
        // 3 — CallStop
        $ext->add($section, $extension, '', new \ext_set(
            'CallStop',
            '${STRFTIME(epoch,,%s)}'
        ));
    
        // 4 — innumber
        $ext->add($section, $extension, '', new \ext_set(
            'innumber',
            '${DB(${FROM_DID}/${CALLERID(num)}/num)}'
        ));
    
        // 5 — CallStart
        $ext->add($section, $extension, '', new \ext_set(
            'CallStart',
            '${DB(${FROM_DID}/${CALLERID(num)}/num/callstart)}'
        ));
    
        // 6 — timecall
        $ext->add($section, $extension, '', new \ext_set(
            'timecall',
            '${MATH(${CallStop}-${CallStart},int)}'
        ));
    
        // 7 — timecalldur
        $ext->add($section, $extension, '', new \ext_set(
            'timecalldur',
            '${MATH(${CallStop}-${CallinStart},int)}'
        ));
    
        // 8 — memrealnum
        $ext->add($section, $extension, '', new \ext_set(
            'memrealnum',
            '${CUT(CUT(MEMBERINTERFACE,/,2),@,1)}'
        ));
    
        // 9 — FMPR override
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${ringbackexten:0:4}" = "FMPR"]',
            'Set',
            'memrealnum=${CONNECTEDLINE(num)}'
        ));
    
        // 10 — NoOp(memrealnum)
        $ext->add($section, $extension, '', new \ext_noop('${memrealnum}'));
    
        // 11 — установка __innumber
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${memrealnum}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/totransfer-queue)}" = "" & "${innumber}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/blindt)}" != "yes" & "${BLKVM_CHANNEL}" = ""]',
            'Set',
            '__innumber=${memrealnum}'
        ));
    
        // 12 — System(hangup.sh)
        $ext->add($section, $extension, 'send', new \ext_execif(
            '$["${innumber}" != "" & "${ZRUCHNA_CLIENT_NAME}" != ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/hangup.sh\' '
            .'"${CALLERID(num)}" "${FROM_DID}" "${innumber}" "${timecall}" "ANSWERED" '
            .'"/var/spool/asterisk/monitor/${YEAR}/${MONTH}/${DAY}/${CALLFILENAME}.${ZRUCHNA_FORMAT_RECORD}" '
            .'"${CHANNEL(LINKEDID)}" "${DB(${FROM_DID}/${CALLERID(num)}/account)}" &'
        ));
    
        // 13 — переход в in-call-end-dbdel-bx24
        $ext->add($section, $extension, '', new \ext_gosubif(
            '$["${innumber}" != ""]',
            'in-call-end-dbdel-bx24,s,1'
        ));
    
        // 14 — NoOp()
        $ext->add($section, $extension, '', new \ext_noop());
    
        // 15 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addInCallEndDbdelContext(\extensions $ext)
    {
        $section = 'in-call-end-dbdel-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — DB_DELETE(num/callstart)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/num/callstart)}'
        ));
    
        // 3 — DB_DELETE(account)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/account)}'
        ));
    
        // 4 — DB_DELETE(queuecount)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queuecount)}'
        ));
    
        // 5 — DB_DELETE(num)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/num)}'
        ));
    
        // 6 — DB_DELETE(dstnum)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/dstnum)}'
        ));

        // 7 — DB_DELETE(dstext)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/dstext)}'
        ));
    
        // 8 — DB_DELETE(queue)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${FROM_DID}/${CALLERID(num)}/queue)}'
        ));
    
        // 9 — DB_DELETE(outtotransfer)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${MEMBERNAME}/outtotransfer)}'
        ));
    
        // 10 — DB_DELETE(outtotransfermiss)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(${MEMBERNAME}/outtotransfermiss)}'
        ));
    
        // 11 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addOuthangupContext(\extensions $ext)
    {
        $section = 'outhangup-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // 1 — NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // 2 — pjsipstatus
        $ext->add($section, $extension, '', new \ext_set(
            'pjsipstatus',
            '${CUT(CDR(dstchannel),/,1)}'
        ));
    
        // 3 — CallStop
        $ext->add($section, $extension, '', new \ext_set(
            'CallStop',
            '${STRFTIME(epoch,,%s)}'
        ));
    
        // 4 — AMPUSERCID override
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${FROMEXTEN:0:1}" = "1"]',
            'Set',
            '_AMPUSERCID=${FROMEXTEN}'
        ));
    
        // 5 — statusdial
        $ext->add($section, $extension, '', new \ext_set(
            '__statusdial',
            '${CDR(disposition)}'
        ));
    
        // 6 — numbclient fallback
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${numbclient}" = ""]',
            'Set',
            '__numbclient=${CALLERID(num)}'
        ));
    
        // 7 — NO ANSWER override
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${statusdial}" = "ANSWERED" & "${DB(out/${AMPUSERCID}/${numbclient}/startcdranswer)}" = "" & "${NODEST}" = ""]',
            'Set',
            '__statusdial=NO ANSWER'
        ));
    
        // 8 — BUSY override
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${DIALSTATUS}" = "BUSY"]',
            'Set',
            '__statusdial=BUSY'
        ));
    
        // 9 — ANSWERED override
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${DIALSTATUS}" = "ANSWER"]',
            'Set',
            '__statusdial=ANSWERED'
        ));
    
        // 10 — startcdranswer
        $ext->add($section, $extension, '', new \ext_set(
            'startcdranswer',
            '${DB(out/${AMPUSERCID}/${numbclient}/startcdranswer)}'
        ));
    
        // 11 — destinationame
        $ext->add($section, $extension, '', new \ext_set(
            'destinationame',
            '${DB(cidname/${numbclient})}'
        ));
    
        // 12 — billsec = 0
        $ext->add($section, $extension, '', new \ext_set(
            'billsec',
            '0'
        ));
    
        // 13 — trunkname detection (case 1)
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" = "" & "${trunkname}" = ""]',
            'Set',
            '__trunkname=${CUT(CUT(CDR(dstchannel),/,2),-,1)}'
        ));
    
        // 14 — trunkname detection (case 2)
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" = "" & "${trunkname}" = "" & "${pjsipstatus}" != "PJSIP"]',
            'Set',
            '__trunkname=${CUT(custom,/,2)}'
        ));
    
        // 15 — trunkname detection (case 3)
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" != "" & "${trunkname}" = "" & "${pjsipstatus}" != "PJSIP"]',
            'Set',
            '__trunkname=${CUT(CUT(post_num,@,2),/,1)}'
        ));
    
        // 16 — trunkname detection (case 4)
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${calldirection}" = "out" & "${CUT(CUT(post_num,@,2),/,1)}" = "" & "${totransfer}" != ""]',
            'Set',
            '__trunkname=${CUT(ringbackexten,/,1)}'
        ));
    
        // 17 — billsec calculation
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${startcdranswer}" != ""]',
            'Set',
            'billsec=${MATH(${CallStop}-${startcdranswer},int)}'
        ));
    
        // 18 — NoOp(trunkname)
        $ext->add($section, $extension, '', new \ext_noop('${trunkname}'));
    
        // 19 — System(hangup2.sh)
        $ext->add($section, $extension, 'ats2bx24-hangup', new \ext_execif(
            '$["${trunkname}" != "" & "${AMPUSERCID}" != "${trunkname}" & "${ZRUCHNA_CLIENT_NAME}" != "" & "${startcdrout}" != ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/hangup2.sh\' '
            .'"${numbclient}" "${trunkname}" "${AMPUSERCID}" "${billsec}" "${statusdial}" '
            .'"/var/spool/asterisk/monitor/${YEAR}/${MONTH}/${DAY}/${CALLFILENAME}.${ZRUCHNA_FORMAT_RECORD}" '
            .'"${UNIQUEID}" "${CHANNEL(accountcode)}" &'
        ));
    
        // 20 — DB_DELETE(startcdranswer)
        $ext->add($section, $extension, '', new \ext_noop(
            '${DB_DELETE(out/${AMPUSERCID}/${numbclient}/startcdranswer)}'
        ));
    
        // 21 — Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addInCallAnswerBx24Context(\extensions $ext)
    {
        $section = 'in-call-answer-bx24';
        $ext->addSectionNoCustom($section, true);
    
        $extension = 's';
    
        // exten => s,1,NoOp()
        $ext->add($section, $extension, '1', new \ext_noop());
    
        // exten => s,n,Set(CallStart=...)
        $ext->add($section, $extension, '', new \ext_setvar(
            'CallStart',
            '${DB(${FROM_DID}/${CALLERID(num)}/num/callstart)}'
        ));
    
        // exten => s,n,ExecIf(...)
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${CallStart}" = ""]',
            'Set',
            'DB(${FROM_DID}/${clientnum}/num/callstart)=${STRFTIME(epoch,,%s)}'
        ));
    
        // exten => s,n,Set(numb=...)
        $ext->add($section, $extension, '', new \ext_setvar(
            'numb',
            '${DB(DEVICE/${CALLERID(num)}/user)}'
        ));
    
        // exten => s,n,ExecIf(...)
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${CallStart}" = "" & "${FMGRP}" != ""]',
            'Set',
            'numb=${FMGRP}'
        ));
    
        // exten => s,n,Set(DB(...)=...)
        $ext->add($section, $extension, '', new \ext_setvar(
            'DB(${FROM_DID}/${clientnum}/num)',
            '${numb}'
        ));
    
        // exten => s,n,Set(AC=...)
        $ext->add($section, $extension, '', new \ext_setvar(
            'AC',
            '${DB(${FROM_DID}/${clientnum}/account)}'
        ));
    
        // exten => s,n,ExecIf($["${AC}"=""]?Set(AC=${variac}))
        $ext->add($section, $extension, '', new \ext_execif(
            '$["${AC}" = ""]',
            'Set',
            'AC=${variac}'
        ));
    
        // exten => s,n(ats2bx24-answer),ExecIf(...)
        // ТЕГ МОЖНО СТАВИТЬ ТОЛЬКО НА add(), НЕ НА splice()
        $ext->add($section, $extension, 'ats2bx24-answer', new \ext_execif(
            '$["${ZRUCHNA_CLIENT_NAME}" != ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/answer.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${FROM_DID}" "${numb}" "${clientnum}" '
            .'"${STRFTIME(${EPOCH},,%d-%m-%Y %H:%M:%S)}" "${AC}" &'
        ));
    
        // exten => s,n,Return()
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function modifyMacroAutoBlkvmContext(\extensions $ext)
    {
        $section = 'macro-auto-blkvm';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }
    
        $extension = 's';
        $extenKey  = ' '.$extension.' ';
    
        if (empty($ext->_exts[$section][$extenKey])) {
            throw new \UnexpectedValueException(sprintf('Section "%s" has no "%s" extension', $section, $extension));
        }
    
        //
        // Ищем строку Set(MASTER_CHANNEL(FORWARD_CONTEXT)=from-internal)
        //
        $insertPos = null;
    
        foreach ($ext->_exts[$section][$extenKey] as $priority => $command) {
    
            if ($command['cmd'] instanceof \ext_setvar) {
    
                $var   = $command['cmd']->var   ?? '';
                $value = $command['cmd']->value ?? '';
    
                if ($var === 'MASTER_CHANNEL(FORWARD_CONTEXT)' &&
                    $value === 'from-internal') {
    
                    $insertPos = $priority + 1;
                    break;
                }
            }
        }
    
        if ($insertPos === null) {
            throw new \UnexpectedValueException(
                'Cannot find Set(MASTER_CHANNEL(FORWARD_CONTEXT)=from-internal) in macro-auto-blkvm'
            );
        }
    
        //
        // Вставляем Gosub(in-call-answer-bx24,s,1)
        //
        $ext->splice(
            $section,
            $extension,
            $insertPos,
            new \ext_gosub('1', 's', 'in-call-answer-bx24')
        );
    }




    private function modifyMacroHangupcallContext(\extensions $ext)
    {
        $section = 'macro-hangupcall';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }
    
        $extension = 's';
        $insertPos = null;
    
        // Ищем строку Hangup()
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_hangup) {
                $insertPos = $priority;
                break;
            }
        }
    
        if ($insertPos === null) {
            throw new \UnexpectedValueException(sprintf(
                'Unable to find Hangup() in %s@%s',
                $extension,
                $section
            ));
        }
    
        // Вставляем GoSubIf перед Hangup()
        $ext->splice($section, $extension, $insertPos, new \ext_gosubif(
            '$["${calldirection}" = "out"]',
            'outhangup-bx24,s,1'
        ));
    }


    private function modifyMacroDialoutTrunk(\extensions $ext)
    {
        $section = 'macro-dialout-trunk';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }
    
        $extension = 's';
        $extenKey = ' '.$extension.' ';
    
        //
        // БЛОК 1 — после Set(DIAL_TRUNK=${ARG1})
        //
        $insertPos1 = null;
    
        foreach ($ext->_exts[$section][$extenKey] as $priority => $command) {
    
            // Ищем ext_setvar, как в старом модуле
            if ($command['cmd'] instanceof \ext_setvar) {
    
                $var   = $command['cmd']->var   ?? '';
                $value = $command['cmd']->value ?? '';
    
                if ($var === 'DIAL_TRUNK' && $value === '${ARG1}') {
                    $insertPos1 = $priority + 1;
                    break;
                }
            }
        }
    
        if ($insertPos1 === null) {
            throw new \UnexpectedValueException('Cannot find Set(DIAL_TRUNK=${ARG1})');
        }
    
        // Вставляем блок 1
        $ext->splice($section, $extension, $insertPos1++, new \ext_agi('accountuniqeout.sh'));
        $ext->splice($section, $extension, $insertPos1++, new \ext_setvar('__calldirection', 'out'));
        $ext->splice($section, $extension, $insertPos1++, new \ext_setvar('__accountcode', '${CHANNEL(accountcode)}'));
        $ext->splice($section, $extension, $insertPos1++, new \ext_execif(
            '$["${CHANNEL(accountcode)}" = ""]',
            'Set',
            'CHANNEL(accountcode)=${accoutout}'
        ));
        $ext->splice($section, $extension, $insertPos1++, new \ext_setvar(
            '__startcdrout',
            '${STRFTIME(epoch,,%s)}'
        ));
    
    
        //
        // БЛОК 2 — перед GotoIf($["${custom}" = "AMP"]?customtrunk)
        //
        $insertPos2 = null;
    
        foreach ($ext->_exts[$section][$extenKey] as $priority => $command) {
    
            if ($command['cmd'] instanceof \ext_gotoif) {
    
                $cond = $command['cmd']->condition ?? '';
    
                if (strpos($cond, '"${custom}" = "AMP"') !== false) {
                    $insertPos2 = $priority;
                    break;
                }
            }
        }
    
        if ($insertPos2 === null) {
            throw new \UnexpectedValueException('Cannot find GotoIf("${custom}" = "AMP")');
        }
    
        // Вставляем блок 2
        $ext->splice($section, $extension, $insertPos2++, new \ext_setvar(
            '__numbclient',
            '${FILTER(+0123456789,${CONNECTEDLINE(num)})}'
        ));
        $ext->splice($section, $extension, $insertPos2++, new \ext_execif(
            '$["${numbclient:0:3}" = "375"]',
            'Set',
            '__numbclient=+375${numbclient:3}'
        ));
        $ext->splice($section, $extension, $insertPos2++, new \ext_execif(
            '$["${numbclient:0:2}" = "80"]',
            'Set',
            '__numbclient=+375${numbclient:2}'
        ));
        $ext->splice($section, $extension, $insertPos2++, new \ext_execif(
            '$["${numbclient:0:2}" = "00"]',
            'Set',
            '__numbclient=+375${numbclient:2}'
        ));
        $ext->splice($section, $extension, $insertPos2++, new \ext_execif(
            '$["${numbclient:0:1}" = "7"]',
            'Set',
            '__numbclient=+7${numbclient:1}'
        ));
        $ext->splice($section, $extension, $insertPos2++, new \ext_execif(
            '$["${ZRUCHNA_CLIENT_NAME}" != "" & "${FMGL_DIAL}" = ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/outdial.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${numbclient}" "${AMPUSERCID}" "${CDR(start)}" "${CHANNEL(accountcode)}" &'
        ));
    
    
        //
        // БЛОК 3 — после Set(the_num=${OUTNUM})
        //
        $insertPos3 = null;
    
        foreach ($ext->_exts[$section][$extenKey] as $priority => $command) {
    
            if ($command['cmd'] instanceof \ext_setvar) {
    
                $var   = $command['cmd']->var   ?? '';
                $value = $command['cmd']->value ?? '';
    
                if ($var === 'the_num' && $value === '${OUTNUM}') {
                    $insertPos3 = $priority + 1;
                    break;
                }
            }
        }
    
        if ($insertPos3 === null) {
            throw new \UnexpectedValueException('Cannot find Set(the_num=${OUTNUM})');
        }
    
        // Вставляем блок 3
        $ext->splice($section, $extension, $insertPos3, new \ext_execif(
            '$["${ZRUCHNA_CLIENT_NAME}" != "" & "${FMGL_DIAL}" = ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/outdial.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${numbclient}" "${AMPUSERCID}" "${CDR(start)}" "${CHANNEL(accountcode)}" &'
        ), 'skipoutnum');
    }



    private function modifyMacroDialOneContext(\extensions $ext)
    {
        $section = 'dialOne-with-exten';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }
    
        $extension = '_[+-X].';
        $extenKey  = ' '.$extension.' ';
    
        if (empty($ext->_exts[$section][$extenKey])) {
            throw new \UnexpectedValueException(sprintf('Section "%s" has no "%s" extension', $section, $extension));
        }
    
        //
        // Ищем строку:
        // exten => _[+-X].,1,Set(CHANNEL(hangup_handler_push)=app-missedcall-hangup,${DialMCEXT},1)
        //
        $insertPos = null;
    
        foreach ($ext->_exts[$section][$extenKey] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_setvar) {
                $var   = $command['cmd']->var   ?? '';
                $value = $command['cmd']->value ?? '';
    
                if ($var === 'CHANNEL(hangup_handler_push)' &&
                    $value === 'app-missedcall-hangup,${DialMCEXT},1') {
                    $insertPos = $priority + 1;
                    break;
                }
            }
        }
    
        if ($insertPos === null) {
            throw new \UnexpectedValueException(
                'Cannot find Set(CHANNEL(hangup_handler_push)=app-missedcall-hangup,${DialMCEXT},1) in dialOne-with-exten'
            );
        }
    
        //
        // Вставляем твой блок после найденной строки
        //
        $p = $insertPos;
    
        // ExecIf($["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" = ""]?Set(DB(${FROM_DID}/${CALLERID(num)}/queue)=${NODEST}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" = ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/queue)=${NODEST}'
        ));
    
        // ExecIf($["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" != "${NODEST}"]?Set(DB(${FROM_DID}/${CALLERID(num)}/queuecount)=0))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" != "${NODEST}"]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=0'
        ));
    
        // ExecIf($["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" != "${NODEST}"]?Set(DB(${FROM_DID}/${CALLERID(num)}/queue)=${NODEST}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${DB(${FROM_DID}/${CALLERID(num)}/queue)}" != "${NODEST}"]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/queue)=${NODEST}'
        ));
    
        // Set(ringallstrat=NO)
        $ext->splice($section, $extension, $p++, new \ext_setvar('ringallstrat', 'NO'));
    
        // NoOp(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)})
        $ext->splice($section, $extension, $p++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'));
    
        // Set(strat=${SHELL(asterisk -rx 'queue show ${NODEST}' | grep ringall | awk '{print$1}' | tr -d '\r\n')})
        $ext->splice($section, $extension, $p++, new \ext_setvar(
            'strat',
            '${SHELL(asterisk -rx \'queue show ${NODEST}\' | grep ringall | awk \'{print$1}\' | tr -d \'\\r\\n\')}'
        ));
    
        // ExecIf($["${strat}" = "${NODEST}"]?Set(ringallstrat=YES))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${strat}" = "${NODEST}"]',
            'Set',
            'ringallstrat=YES'
        ));
    
        // NoOp("${strat}" = "${NODEST}")
        $ext->splice($section, $extension, $p++, new \ext_noop('"${strat}" = "${NODEST}"'));
    
        // ExecIf($["${ringallstrat}" = "YES"]?Set(dstext=${FILTER(,0123456789,${QUEUE_MEMBER_LIST(${NODEST})})}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${ringallstrat}" = "YES"]',
            'Set',
            'dstext=${FILTER(0123456789\,,${QUEUE_MEMBER_LIST(${NODEST})})}'
        ));
    
        // ExecIf($["${ringallstrat}" != "YES"]?Set(memcount=${QUEUE_MEMBER_COUNT(${NODEST})}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${ringallstrat}" != "YES"]',
            'Set',
            'memcount=${QUEUE_MEMBER_COUNT(${NODEST})}'
        ));
    
        // NoOp(...)
        $ext->splice($section, $extension, $p++, new \ext_noop(
            '${FROMQ} ${AGENT_DEST} FROMQ=${FROMQ} strategy=${ringallstart} Nodest=${NODEST} '
            .'Members=${QUEUE_MEMBER_LIST(${NODEST})} Members2=${FILTER(0123456789\,,${QUEUE_MEMBER_LIST(${NODEST})})} '
            .'membercount=${QUEUE_MEMBER_COUNT(${NODEST})} memberlist=${QUEUE_MEMBER_LIST(${NODEST})})'
        ));
    
        // ExecIf(... queue.sh ... dstext ...)
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}" < 1 & "${FMGL_DIAL}" = ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/queue.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${clientnum}" "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}" '
            .'"${CHANNEL(accountcode)}" "${dstext}" &'
        ));
    
        // ExecIf(... dstext != "" ? queuecount++)
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" != "" ]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=${MATH(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}+1,int)}'
        ));
    
        // NoOp(queuecount)
        $ext->splice($section, $extension, $p++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'));
    
        // ExecIf(... dstext = "" & queuecount < memcount ? queuecount++)
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = "" & ${DB(${FROM_DID}/${CALLERID(num)}/queuecount)} < ${memcount}]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=${MATH(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}+1,int)}'
        ));
    
        // NoOp(queuecount)
        $ext->splice($section, $extension, $p++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'));
    
        // ExecIf(... ringallstrat != YES & memcount = "" & queuecount < 1 ? queue.sh ... ARG3)
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = "" & "${ringallstrat}" != "YES" & "${memcount}" = "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}" < 1 & "${FMGL_DIAL}" = ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/queue.sh\' '
            .'"${CHANNEL(LINKEDID)}" "${clientnum}" "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}" '
            .'"${CHANNEL(accountcode)}" "${ARG3}" &'
        ));
    
        // ExecIf(... ringallstrat != YES & memcount = "" ? queuecount++)
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = "" & "${ringallstrat}" != "YES" & "${memcount}" = ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/queuecount)=${MATH(${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}+1,int)}'
        ));

        // ExecIf($["${clientnum}" != "" & "${dstext}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}" < 1 & "${FMGL_DIAL}" = ""]?Set(DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${CUT(dstext,\,,1)}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}" < 1 & "${FMGL_DIAL}" = ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${CUT(dstext,\,,1)}'
        ));

        // ExecIf($["${clientnum}" != "" & "${dstext}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}" < 1 & "${FMGL_DIAL}" = ""]?Set(DB(${FROM_DID}/${CALLERID(num)}/dstext)=${dstext}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" != "" & "${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}" < 1 & "${FMGL_DIAL}" = ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/dstext)=${dstext}'
        ));

        // NoOp(queuecount)
        $ext->splice($section, $extension, $p++, new \ext_noop('${DB(${FROM_DID}/${CALLERID(num)}/queuecount)}'));
    
        // NoOp(${LEN(${CALLERID(num)})})
        $ext->splice($section, $extension, $p++, new \ext_noop('${LEN(${CALLERID(num)})}'));
    
        // Set(DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${ARG3})
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${dstext}" = ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${ARG3}'
        ));
    
        // ExecIf($["${calldirection}" != "out" & "${totransfer}" != ""]?Set(DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${totransfer}))
        $ext->splice($section, $extension, $p++, new \ext_execif(
            '$["${calldirection}" != "out" & "${totransfer}" != ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${totransfer}'
        ));
    }

    private function modifyMacroDialContext(\extensions $ext)
    {
        $section = 'macro-dial';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }
    
        $extension = 's';
        $insertPos = null;
    
        // Ищем строку NoOp(Working with ${EXTTOCALL})
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_noop &&
                strpos($command['cmd']->data, 'Working with ${EXTTOCALL}') !== false) {
                $insertPos = $priority + 1;
                break;
            }
        }
    
        if ($insertPos === null) {
            throw new \UnexpectedValueException('Cannot find NoOp(Working with ${EXTTOCALL}) in macro-dial');
        }
    
        //
        // Вставляем блок строго по эталону
        //
    
        // 1 — innubdial = ARG3 (если нет "-")
        $ext->splice($section, $extension, $insertPos++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = "" & "${CUT(ARG3,-,2)}" = ""]',
            'Set',
            'innubdial=${ARG3}'
        ));
    
        // 2 — innubdial = CUT(ARG3,-,1) (если есть "-")
        $ext->splice($section, $extension, $insertPos++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = "" & "${CUT(ARG3,-,2)}" != ""]',
            'Set',
            'innubdial=${CUT(ARG3,-,1)}'
        ));
    
        // 3 — System(queue.sh)
        $ext->splice($section, $extension, $insertPos++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = "" & "${ZRUCHNA_CLIENT_NAME}" != "" & "${FMGL_DIAL}" = ""]',
            'System',
            'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/queue.sh\' '
            .' "${CHANNEL(LINKEDID)}" "${clientnum}" "${STRFTIME(${EPOCH},,%d-%m-%Y%H:%M:%S)}" '
            .' "${CHANNEL(accountcode)}" "${ARG3}" &'
        ));
    
        // 4 — DB(dstnum) = innubdial
        $ext->splice($section, $extension, $insertPos++, new \ext_execif(
            '$["${clientnum}" != "" & "${dstext}" = ""]',
            'Set',
            'DB(${FROM_DID}/${CALLERID(num)}/dstnum)=${innubdial}'
        ));
    }



    private function modifyMacroDialoutOneContext(\extensions $ext)
    {
        $section = 'macro-dialout-one';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }
    
        $extension = 's';
    
        //
        // 1. Найти место после строки:
        //    Set(CONNECTEDLINE(num,i)=${CALLERID(num)})
        //
        $afterPriority = null;
    
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ($command['cmd'] instanceof \ext_set) {
                if ($command['cmd']->var === 'CONNECTEDLINE(num,i)') {
                    $afterPriority = $priority + 1;
                    break;
                }
            }
        }
    
        if ($afterPriority === null) {
            throw new \UnexpectedValueException(sprintf(
                'Unable to find Set(CONNECTEDLINE(num,i)=...) in %s@%s',
                $extension,
                $section
            ));
        }
    
        //
        // 2. Вставляем обработку ответа на исходящий звонок
        //
    
        // numbclient = CONNECTEDLINE(num)
        $ext->splice(
            $section,
            $extension,
            $afterPriority++,
            new \ext_set('numbclient', '${CONNECTEDLINE(num)}')
        );
    
        // нормализация номера
        $ext->splice(
            $section,
            $extension,
            $afterPriority++,
            new \ext_execif(
                '$["${numbclient:0:3}" = "375"]',
                'Set',
                'numbclient=+375${numbclient:3}'
            )
        );
        $ext->splice(
            $section,
            $extension,
            $afterPriority++,
            new \ext_execif(
                '$["${numbclient:0:2}" = "80"]',
                'Set',
                'numbclient=+375${numbclient:2}'
            )
        );
        $ext->splice(
            $section,
            $extension,
            $afterPriority++,
            new \ext_execif(
                '$["${numbclient:0:2}" = "00"]',
                'Set',
                'numbclient=+375${numbclient:2}'
            )
        );
        $ext->splice(
            $section,
            $extension,
            $afterPriority++,
            new \ext_execif(
                '$["${numbclient:0:1}" = "7"]',
                'Set',
                'numbclient=+7${numbclient:1}'
            )
        );
    
        // System(outanswer.sh)
        if ('' !== $this->getClientName()) {
            $ext->splice(
                $section,
                $extension,
                $afterPriority++,
                new \ext_execif(
                    '$["${ZRUCHNA_CLIENT_NAME}" != ""]',
                    'System',
                    'klient="${ZRUCHNA_CLIENT_NAME}" /bin/bash \'/var/www/html/admin/modules/zruchnaio/bin/outanswer.sh\' '
                    .' "${CHANNEL(LINKEDID)}"'
                    .' "${CONNECTEDLINE(num)}"'
                    .' "${CALLERID(num)}"'
                    .' "${CDR(start)}"'
                    .' "${CHANNEL(accountcode)}"'
                    .' &'
                )
            );
        }
    
        //
        // 3. Добавляем вызов outdialanswer-bx24 при ANSWER
        //
        $ext->splice(
            $section,
            $extension,
            $afterPriority++,
            new \ext_gosubif(
                '$["${DIALSTATUS}" = "ANSWER"]',
                'outdialanswer-bx24,s,1'
            )
        );
    }



    /**
     * @param string $binFileName
     * @param array<string, scalar> $environments
     * @return string
     */
    private function getBinFileExec($binFileName, array $environments = array())
    {
        $prefix = '';
        foreach ($environments as $name => $value) {
            $prefix .= sprintf('%s="%s" ', $name, strtr($value, array('"' => '\\"')));
        }

        return $prefix.'/bin/bash '.$this->escapeExtData(escapeshellarg(__DIR__.'/bin/'.$binFileName));
    }

    /**
     * @return string
     */
    protected function getClientName()
    {
        return $this->getConfig('client_name');
    }

    protected function getRecordFormat()
    {
        return $this->getConfig('records_extension');
    }

    private function escapeExtData($value)
    {
        return strtr($value, array(
            ';' => '\;',
            ')' => '\)',
        ));
    }

    public function doConfigPageInit($page)
    {
    }

    public function getActionBar($request)
    {
        $buttons = array();
        switch ($_GET['display']) {
            case 'zruchnaio':
                $buttons = array(
                    'delete' => array(
                        'name' => 'delete',
                        'id' => 'delete',
                        'value' => _('Delete')
                    ),
                    'reset' => array(
                        'name' => 'reset',
                        'id' => 'reset',
                        'value' => _('Reset')
                    ),
                    'submit' => array(
                        'name' => 'submit',
                        'id' => 'submit',
                        'value' => _('Submit')
                    )
                );
                if (empty($_GET['extdisplay'])) {
                    unset($buttons['delete']);
                }
                break;
        }
        return $buttons;
    }

    public function showPage()
    {
        $view = empty($_REQUEST['view']) ? 'main' : $_REQUEST['view'];

        switch ($view) {
            case 'setup':
                switch ($_SERVER['REQUEST_METHOD']) {
                    case 'GET':
                        return load_view(__DIR__ . '/views/setup.php', [
                            'freepbxBaseUrl' => $this->getConfig('freepbx_base_uri'),
                            'recordsExtension' => $this->getConfig('records_extension'),
                            'freepbxCallbackExtension' => $this->getConfig('freepbx_callback_extension'),
                            'clientName' => $this->getClientName(),
                            'error' => isset($_GET['error']) ? $_GET['error'] : '',
                        ]);
                    case 'POST':
                        $this->setConfig('records_extension', $_REQUEST['records_extension']);

                        $_SERVER['REQUEST_URI'] = preg_replace(
                            ['~error(?:=[^&#]*)?~', '~&{2,}~'],
                            '&',
                            $_SERVER['REQUEST_URI']
                        );

                        if (!empty($_REQUEST['bx24_application_secret'])) {
                            if ($this->touchIntegrationConfigs($_REQUEST['bx24_application_secret'], $_REQUEST['freepbx_base_uri'], $_REQUEST['freepbx_callback_extension'])) {
                                $this->setConfig('freepbx_base_uri', $_REQUEST['freepbx_base_uri']);
                                $this->setConfig('freepbx_callback_extension', $_REQUEST['freepbx_callback_extension']);
                            } else {
                                $_SERVER['REQUEST_URI'] .= false !== strpos($_SERVER['REQUEST_URI'], '?') ? '&' : '?';
                                $_SERVER['REQUEST_URI'] .= 'error=failed-public-url';
                            }
                        }

                        \needreload();

                        header('Location: '.$_SERVER['REQUEST_URI'], true, 302);
                        return '';
                }
            default:
                $vars = array('helloworld' => _("Hello World"));
                return load_view(__DIR__ . '/views/main.php', $vars);
        }
    }

    /**
     * @param string $bx24Secret
     * @param string $freepbxBaseUri
     * @param string $freepbxCallbackExtension
     * @throws \RuntimeException
     * @return bool
     *
     * @api
     */
    private function touchIntegrationConfigs($bx24Secret, $freepbxBaseUri, $freepbxCallbackExtension)
    {
        $ch = curl_init('https://asterisk-connect.zruchna.io/freepbx/api/v1/registration');
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_CUSTOMREQUEST => 'PUT',
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/x-www-form-urlencoded',
            ],
            CURLOPT_POSTFIELDS => http_build_query([
                'registration_info' => [
                    'freepbx_module_secret' => $this->getSecretValue(),
                    'bitrix24_app_secret' => $bx24Secret,
                    'freepbx_base_uri' => $freepbxBaseUri,
                    'freepbx_callback_extension' => $freepbxCallbackExtension,
                ],
            ], '', '&', PHP_QUERY_RFC3986),
        ]);
        $responseBody = curl_exec($ch);
        $statusCode = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        $errorCode = curl_errno($ch);
        $errorMessage = curl_error($ch);
        curl_close($ch);

        if (CURLE_OK !== $errorCode) {
            throw new \RuntimeException('Unable to touch integration config: cURL ['.$errorCode.'] '.$errorMessage);
        }

        $responsePayload = json_decode($responseBody, true);

        if ($statusCode < 200 || $statusCode >= 300) {
            throw new \RuntimeException('Unable to touch integration config: ['.$statusCode.']: '.$responsePayload['message']);
        }

        if (isset($responsePayload['clientName'])
            && is_string($responsePayload['clientName'])
        ) {
            $this->setConfig('client_name', $responsePayload['clientName']);
        }

        return $responsePayload['status'];
    }

    /**
     * @param string $req
     * @param array $setting
     * @return bool
     *
     * @noinspection PhpUnused
     */
    public function ajaxRequest($req, &$setting)
    {
        switch ($req) {
            case 'ping':
            case 'is_signed':
                $this->allowAnonymous($setting);

                return true;
            case 'originate':
                if ('POST' !== $_SERVER['REQUEST_METHOD']) {
                    return false;
                }

                $this->allowAnonymous($setting);

                if (!isset($_GET['secret'])) {
                    header('X-Reject-Reason: Miss required parameter "secret"', false);

                    return false;
                }
                if ($this->getSecretValue() !== $_GET['secret']) {
                    header('X-Reject-Reason: Invalid value of "secret" parameter', false);

                    // To prevent bruteforce.
                    sleep(2);

                    return false;
                }

                return true;
            case 'record':
                if ('GET' !== $_SERVER['REQUEST_METHOD']) {
                    header('X-Reject-Reason: Non-supported request method', false);

                    return false;
                }

                $this->allowAnonymous($setting);

                if (!isset($_GET['secret'])) {
                    header('X-Reject-Reason: Miss required parameter "secret"', false);

                    return false;
                }
                if ($this->getSecretValue() !== $_GET['secret']) {
                    header('X-Reject-Reason: Invalid value of "secret" parameter', false);

                    // To prevent bruteforce.
                    sleep(2);

                    return false;
                }
                if (!isset($_GET['call_id'])) {
                    header('X-Reject-Reason: Miss required parameter "call_id"', false);

                    return false;
                }
                if (!\is_string($_GET['call_id'])) {
                    header(sprintf('X-Reject-Reason: Expected to "call_id" parameter be string, %s provided', gettype($_GET['call_id'])), false);

                    return false;
                }

                return true;
            default:
                return false;
        }
    }

    public function ajaxHandler()
    {
        switch ($_REQUEST['command']) {
            case 'ping':
                return 'pong';
            case 'is_signed':
                if (!isset($_GET['secret'])) {
                    header('X-Reject-Reason: Miss required parameter "secret"', false);

                    return 'no';
                }

                // To prevent bruteforce.
                sleep(2);

                if ($this->getSecretValue() !== $_GET['secret']) {
                    header('X-Reject-Reason: Invalid value of "secret" parameter', false);

                    return 'no';
                }

                return 'yes';
            case 'originate':
                $extension = isset($_POST['extension']) ? $_POST['extension'] : null;
                $phone = isset($_POST['phone']) ? $_POST['phone'] : null;
                $callId = isset($_POST['call_id']) ? $_POST['call_id'] : null;
                $lineNumber = isset($_POST['line_number']) ? $_POST['line_number'] : null;

                if (empty($extension)) {
                    return false;
                }
                if (empty($phone)) {
                    return false;
                }
                if (empty($callId)) {
                    return false;
                }

                $callIdShort = substr($callId, 0, 20);
                $response = $this->astman->Originate([
                    'Channel' => sprintf('Local/%s@from-internal', $extension),
                    'CallerID' => sprintf('%s', $extension),
                    'Timeout' => '20000',
                    'Account' => sprintf('%s', $callIdShort),
                    'Context' => 'from-internal',
                    'Exten' => sprintf('%s', $phone),
                    'Priority' => '1',
                    'Variable' => [
                        'variac' => sprintf('%s', $callId),
                        'line' => sprintf('%s', $lineNumber),
                        'calldirection' => 'out',
                    ],
                ]);

                $response = array_change_key_case($response, \CASE_LOWER);
                if (strtolower($response['response']) === 'error') {
                    return sprintf('Error: %s', $response['message']);
                }

                return 'Done';
            case 'record':
                $callId = isset($_GET['call_id']) ? $_GET['call_id'] : null;
                if (!preg_match('~^[1-9]\d*[._]\d+$~', $callId)) {
                    header('Content-Type: application/json', true, 400);
                    echo '{"status":false,"message":"Invalid call ID value provided"}';
                    exit;
                }

                $cdr = \FreePBX::Cdr();
                \assert($cdr instanceof Cdr);

                if ([] === $cdrRecord = $cdr->getRecordByID($callId)) {
                    header('Content-Type: application/json', true, 404);
                    echo '{"status":false,"message":"Unable to find CDR record"}';
                    exit;
                }
                if (empty($cdrRecord['uniqueid'])) {
                    header('Content-Type: application/json', true, 404);
                    echo '{"status":false,"message":"Unable to find CDR record"}';
                    exit;
                }

                $recordFile = $cdrRecord['recordingfile'];
                if (empty($recordFile)) {
                    header('Content-Type: application/json', true, 404);
                    echo '{"status":false,"message":"CDR record does not contain \"recordingfile\" value"}';
                    exit;
                }

                $callId = strtr($callId, '_', '.');

                $recordFileMime = mime_content_type($recordFile) ?: 'application/octet-stream';
                $recordFileSize = filesize($recordFile);
                $recordFileSafeFilename = sprintf('"record %s.%s"', $callId, pathinfo($recordFile, PATHINFO_EXTENSION));
                $recordFileRfc5987Filename = sprintf('UTF-8\'\'%s', rawurlencode(pathinfo($recordFile, PATHINFO_BASENAME)));

                session_write_close();

                while (ob_get_level()) {
                    @ob_end_clean();
                }

                header_remove();
                header(sprintf('Content-Type: %s', $recordFileMime));
                header(sprintf('Content-Length: %d', $recordFileSize));
                header(sprintf('Content-Disposition: attachment; filename=%s; filename*=%s', $recordFileSafeFilename, $recordFileRfc5987Filename));
                header('Cache-Control: private, max-age=2592000'); // =60×60×24×30 (1 month)
                header(sprintf('Expires: %s', \gmdate('D, d M Y H:i:s \G\M\T', strtotime('now +1 month'))));
                header('Connection: close');

                readfile($recordFile);
                exit;
            default:
                return false;
        }
    }

    public function getRightNav($request)
    {
        return <<<'HTML'
<div class="bootstrap-table">
  <div class="fixed-table-toolbar">
    <div class="bs-bars">
      <div class="toolbar-cbbnav">
        <a href="?display=zruchnaio&view=setup" class="btn btn-default">Setup</a>
      </div>
    </div>
  </div>
</div>
HTML;
    }

    /**
     * @return void
     */
    private function allowAnonymous(array &$setting)
    {
        $setting['allowremote'] = true;
        $setting['authenticate'] = false;
    }

    /**
     * @return string|null
     */
    private function getSecretValue()
    {
        try {
            return $this->getConfig('freepbx_module_secret') ?: null;
        } catch (\Exception $e) {
            throw new \LogicException('Impossible error: '.$e->getMessage(), 0, $e);
        }
    }
}
