<?php

namespace Zruchna\FreePbx\RecordsUploader\Proxy;

class UploadingFileNameProxy
{
    /**
     * @var string
     */
    private $rawFilename;

    /**
     * @var string
     */
    private $relatedPath;

    /**
     * @var string|null
     */
    private $baseFilename = null;

    /**
     * @param string $filename
     * @param string $relatedPath
     */
    public function __construct($filename, $relatedPath = 'uploaded/')
    {
        if (!\is_string($filename)) {
            throw new \InvalidArgumentException(sprintf('Expected to $filename be string, %s provided', is_object($filename) ? get_class($filename) : gettype($filename)));
        }

        $this->rawFilename = $filename;
        $this->relatedPath = $relatedPath;
    }

    /**
     * @return string
     */
    public function getRawFilename()
    {
        return $this->rawFilename;
    }

    /**
     * @return string
     */
    public function getBaseFilename()
    {
        if (null !== $this->baseFilename) {
            return $this->baseFilename;
        }

        // "Hello, World!" => "Hello-World-"
        $safeFilename = preg_replace('~\W+~','-', trim($this->rawFilename));

        // "Hello-World-" => "Hello-World"
        $trimmedFilename = trim($safeFilename, '-');

        // "Hello-World" => "hello-world"
        $casedFilename = preg_replace_callback('~(?<=[A-Za-z])(?<char>[A-Z])~', static function (array $matches) {
            return '-'.strtolower($matches['char']);
        }, $trimmedFilename);

        $deduppedFilename = preg_replace('~-{2,}~', '-', $casedFilename);

        return $this->baseFilename = $deduppedFilename;
    }

    /**
     * @return string
     */
    public function getRelatedFilePath()
    {
        return $this->relatedPath.$this->getBaseFilename();
    }

    /**
     * @param string $fileExtension
     * @return string
     */
    public function getTargetFilePath($fileExtension)
    {
        return $this->getRelatedFilePath().'.'.$fileExtension;
    }

    /**
     * @param string $tempDirPath
     * @return string
     */
    public function getTemporaryFilePath($tempDirPath)
    {
        return sprintf('%s/%s', rtrim($tempDirPath, '/'), md5(uniqid($this->getTargetFilePath('tmp'))));
    }
}
