<?php

namespace Zruchna\FreePbx\RecordsUploader\Doctrine\DBAL\Type;

use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Types\ConversionException;
use Doctrine\DBAL\Types\Type;

class DatetimeImmutableUtcType extends Type
{
    const DATETIME_IMMUTABLE_UTC = 'datetime_immutable_utc';

    /** @var \DateTimeZone|null */
    private static $utc = null;

    public function getName()
    {
        return self::DATETIME_IMMUTABLE_UTC;
    }

    /**
     * @param \DateTimeImmutable $value
     * @param AbstractPlatform $platform
     * @return string|null
     */
    public function convertToDatabaseValue($value, AbstractPlatform $platform)
    {
        if (null === $value) {
            return null;
        }

        return $value
            ->setTimezone(self::utc())
            ->format($platform->getDateTimeFormatString());
    }

    /**
     * @param string|null $value
     * @param AbstractPlatform $platform
     * @throws ConversionException
     * @return \DateTimeImmutable|null
     */
    public function convertToPHPValue($value, AbstractPlatform $platform)
    {
        if (null === $value || $value instanceof \DateTimeImmutable) {
            return $value;
        }

        $val = \DateTimeImmutable::createFromFormat('!'.$platform->getDateTimeFormatString(), $value, self::utc());
        if (false === $val) {
            throw ConversionException::conversionFailedFormat($value, $this->getName(), $platform->getDateTimeFormatString());
        }

        return $val;
    }

    public function getSQLDeclaration(array $fieldDeclaration, AbstractPlatform $platform)
    {
        return $platform->getDateTimeTypeDeclarationSQL($fieldDeclaration);
    }

    /**
     * @return \DateTimeZone
     */
    private static function utc()
    {
        if (null === self::$utc) {
            self::$utc = new \DateTimeZone('UTC');
        }

        return self::$utc;
    }
}
