<?php

namespace Zruchna\FreePbx\RecordsUploader\Command;

use Doctrine\ORM\EntityManagerInterface;
use FreePBX\Media;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\StyleInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Zruchna\FreePbx\RecordsUploader\Clock\ClockInterface;
use Zruchna\FreePbx\RecordsUploader\Entity\Recording;
use Zruchna\FreePbx\RecordsUploader\Entity\UploadingItem;
use Zruchna\FreePbx\RecordsUploader\Factory\ClockFactory;
use Zruchna\FreePbx\RecordsUploader\Factory\Doctrine\EntityManagerFactory;
use Zruchna\FreePbx\RecordsUploader\Factory\Repository\SimpleRepositoryFactory;
use Zruchna\FreePbx\RecordsUploader\Proxy\UploadingFileNameProxy;
use Zruchna\FreePbx\RecordsUploader\Repository\RecordingRepository;
use Zruchna\FreePbx\RecordsUploader\Repository\UploadingItemRepository;

class UploadRecordsCommand extends Command
{
    protected static $defaultName = 'upload-records';

    /**
     * @var EntityManagerInterface|null
     */
    private $entityManager = null;

    /**
     * @var UploadingItemRepository|null
     */
    private $uploadingItemRepository = null;

    /**
     * @var ClockInterface|null
     */
    private $clock = null;

    /**
     * @var (OutputInterface&StyleInterface)|null
     */
    private $io = null;

    /**
     * @var Media|null
     */
    private $mediaModule = null;

    /**
     * @var \Recordings|null
     */
    private $recordingsModule = null;

    /**
     * @var RecordingRepository|null
     */
    private $recordingRepository = null;

    /**
     * @var string|null
     */
    private $tempDirPath = null;

    /**
     * @return void
     */
    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->entityManager) {
            $this->entityManager = EntityManagerFactory::getEntityManager();
        }
        if (null === $this->uploadingItemRepository) {
            $this->uploadingItemRepository = SimpleRepositoryFactory::getRepository(UploadingItem::class);
        }
        if (null === $this->clock) {
            $this->clock = ClockFactory::getClock();
        }
        if (null === $this->io) {
            $this->io = new SymfonyStyle($input, $output);
        }
        if (null === $this->mediaModule) {
            $this->mediaModule = \FreePBX::Media();
        }
        if (null === $this->recordingsModule) {
            $this->recordingsModule = \FreePBX::Recordings();
        }
        if (null === $this->recordingRepository) {
            $this->recordingRepository = SimpleRepositoryFactory::getRepository(Recording::class);
        }
        if (null === $this->tempDirPath) {
            $this->tempDirPath = \FreePBX::Config()->get("ASTSPOOLDIR").'/tmp';
        }
    }

    /**
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $now = $this->clock->now();

        $queuedUploadings = $this->uploadingItemRepository->findPending($now);

        if (!$queuedUploadings) {
            if (!$output->isQuiet()) {
                $this->io->note(sprintf('Queued record uploadings at %s was not found', $now->format('Y-m-d H:i:s (P)')));
            }

            return 0;
        }

        if (!$output->isQuiet()) {
            $this->io->writeln(sprintf('Found %d queued record uploadings at %s date', \count($queuedUploadings), $now->format('Y-m-d H:i:s (P)')));
        }

        foreach ($queuedUploadings as $uploadingItem) {
            try {
                $this->uploadRecord($uploadingItem);
            } catch (\Exception $e) {
                $this->io->warning('Unable to upload record: '.$e->getMessage());
                continue;
            }
        }

        return 0;
    }

    /**
     * @param string $targetExtension
     * @return void
     */
    protected function uploadRecord(UploadingItem $uploadingItem, $targetExtension = 'wav')
    {
        $filename = new UploadingFileNameProxy(
            $uploadingItem->getFilename(),
            sprintf('uploaded/%d-', $uploadingItem->getId())
        );

        $temporaryFilePath = $filename->getTemporaryFilePath($this->tempDirPath);

        try {
            $this->downloadRecord($uploadingItem->getUrl(), $temporaryFilePath);
            $uploadingItem->setUploadedAt($this->clock->now());

            $recording = $this->createRecording($uploadingItem);
            $uploadingItem->setRecording($recording);

            $this->convertRecording($temporaryFilePath, $filename->getTargetFilePath($targetExtension));
            $uploadingItem->setConvertedAt($this->clock->now());
        } finally {
            @unlink($temporaryFilePath);
        }

        $recording->setFilename($filename->getRelatedFilePath());

        $this->entityManager->flush();
    }

    /**
     * @throws \RuntimeException
     * @return string
     */
    protected function downloadRecord($sourceUrl, $targetFilePath)
    {
        $targetFileStream = fopen($targetFilePath, 'wb');

        $ch = curl_init($sourceUrl);
        curl_setopt_array($ch, array(
            CURLOPT_FILE => $targetFileStream,
        ));
        curl_exec($ch);

        $responseCode = curl_getinfo($ch, CURLINFO_RESPONSE_CODE);
        $errorCode = curl_errno($ch);
        $errorMessage = curl_error($ch);

        curl_close($ch);
        fclose($targetFileStream);

        if (CURLE_OK !== $errorCode) {
            @unlink($targetFilePath);

            throw new \RuntimeException(sprintf('Unable to download recording: [cURL %d] %s', $errorCode, $errorMessage), $errorCode);
        }

        if (200 > $responseCode || 300 <= $responseCode) {
            @unlink($targetFilePath);

            throw new \RuntimeException(sprintf('Unable to download recording: [HTTP %d] Unexpected response code provided', $responseCode));
        }

        return $targetFilePath;
    }

    /**
     * @throws \RuntimeException
     * @return Recording
     */
    protected function createRecording(UploadingItem $uploadingItem)
    {
        $recordingId = $this->recordingsModule->addRecording(
            $uploadingItem->getFilename(),
            'Uploaded by records_uploader module.',
            '',
            0,
            '',
            'en'
        );

        $recording = $this->recordingRepository->find($recordingId);

        if (null === $recording) {
            throw new \RuntimeException('Unable to create recording');
        }

        return $recording;
    }

    protected function convertRecording($sourcePath, $targetName)
    {
        $recordingsPath = $this->recordingsModule->getPath();
        $targetPath = sprintf('%s/en/%s', $recordingsPath, $targetName);

        if (!file_exists(dirname($targetPath))) {
            mkdir(dirname($targetPath));
        }
        if (!is_dir(dirname($targetPath))) {
            throw new \RuntimeException(sprintf('Expected to "%s" be a directory', dirname($targetPath)));
        }

        try {
            $this->mediaModule->load($sourcePath);
        } catch (\Exception $e) {
            throw new \RuntimeException('Unable to load media file: '.$e->getMessage(), 0, $e);
        }

        $this->mediaModule->convert($targetPath);
    }
}
