<?php

namespace FreePBX\Modules;

use FreePBX\Freepbx_conf;

class Calltracking extends \FreePBX_Helpers implements \BMO
{
    /** @var \FreePBX $FreePBX */
    protected $FreePBX;

    /** @var \DB */
    protected $db;

    /** @var Freepbx_conf */
    private $freePbxConf;

    /**
     * @var \AGI_AsteriskManager
     */
    private $astman;

    /**
     * @param \FreePBX $freepbx
     * @throws \Exception
     * @throws \BadMethodCallException
     */
    public function __construct($freepbx = null)
    {
        parent::__construct($freepbx);

        if (null === $freepbx) {
            throw new \BadMethodCallException('Not given a FreePBX Object');
        }

        $this->FreePBX = $freepbx;
        $this->db = $freepbx->Database;
        $this->freePbxConf = Freepbx_conf::create();
        $this->astman = $freepbx->astman;
    }

    /**
     * Set Priority for Dialplan Hooking
     * Core sits at a priority of 600
     * @method myDialplanHooks
     *
     * @return string        Priority
     */
    public static function myDialplanHooks()
    {
        return 910;
    }

    public function install()
    {
    }

    public function uninstall()
    {
        $this->eraseConfigs();
    }

    /**
     * @throws \InvalidArgumentException
     * @return void
     */
    private function assertConfigs(array $configs)
    {
        $didNamesSet = array();

        foreach ($configs as $idx => $config) {
            if (!\is_array($config)) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be arrays, %s provided at position %s', is_object($config) ? gettype($configs) : get_class($config), $idx));
            }
            if (!isset($config['client'])) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the "client" field, field not found at position %s', $idx));
            }
            if (!\is_string($config['client'])) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the string "client" field, %s provided at position %s', is_object($config['client']) ? gettype($config['client']) : get_class($config['client']), $idx));
            }
            if (!isset($config['secret'])) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the "secret" field, field not found at position %s', $idx));
            }
            if (!\is_string($config['secret'])) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the string "secret" field, %s provided at position %s', is_object($config['secret']) ? gettype($config['secret']) : get_class($config['secret']), $idx));
            }
            if (!isset($config['dids'])) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the "dids" field, field not found at position %s', $idx));
            }
            if (!\is_array($config['dids'])) {
                throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the string "dids" field, %s provided at position %s', is_object($config['dids']) ? gettype($config['dids']) : get_class($config['dids']), $idx));
            }

            foreach ($config['dids'] as $didIdx => $did) {
                if (!\is_string($did)) {
                    throw new \InvalidArgumentException(sprintf('Expected to $configs items be contains the string list "dids" field, %s item type provided at position %s.dids.%s', $did, $idx, $didIdx));
                }

                $did = trim($did);

                if (isset($didNamesSet[$did])) {
                    throw new \InvalidArgumentException(sprintf('Expected to dids be unique, %s did is non-unique, it was already provided on %s client', $did, $didNamesSet[$did]));
                }

                $didNamesSet[$did] = $config['client'];
            }
        }
    }

    /**
     * @return void
     */
    private function eraseConfigs()
    {
        $clientNameList = $this->getClientsConfiguration();
        $didNameList = $this->getDidsConfiguration();

        foreach ($clientNameList as $clientName) {
            $this->astman->database_del($clientName, 'key');
        }
        foreach ($didNameList as $didName) {
            $this->astman->database_del('calltrack', $didName);
        }

        $this->astman->database_deltree('zruchnaio/calltracking');
    }

    /**
     * @param list<array{client: string, secret: string|null, dids: list<string>}> $configs
     * @return void
     */
    private function applyConfigs(array $configs)
    {
        if (array() === $configs) {
            return;
        }

        $clientNameList = array();
        $didNameSet = array();
        foreach ($configs as $config) {
            $clientNameList[] = $config['client'];

            $this->astman->database_put($config['client'], 'key', $config['secret']);

            foreach ($config['dids'] as $did) {
                $didNameSet[$did] = true;

                $this->astman->database_put('calltrack', $did, $config['client']);
            }
        }

        $this->astman->database_put('zruchnaio/calltracking', 'configured', gmdate(\DATE_ATOM));
        $this->putSequenceDatabaseValue('zruchnaio/calltracking/clients', $clientNameList);
        $this->putSequenceDatabaseValue('zruchnaio/calltracking/dids', array_keys($didNameSet));
    }

    /**
     * @return list<array{client: string, secret: string|null, dids: list<string>}>
     */
    private function getConfigs()
    {
        if (empty($this->astman->database_get('zruchnaio/calltracking', 'configured'))) {
            return array();
        }

        $clientNameList = $this->getClientsConfiguration();
        $didNameList = $this->getDidsConfiguration();

        $clientDidsMap = array();
        foreach ($didNameList as $didName) {
            $clientName = $this->astman->database_get('calltrack', $didName);
            if (empty($clientName)) {
                continue;
            }

            $clientDidsMap[$clientName][] = $didName;
        }

        $configs = array();
        foreach ($clientNameList as $clientName) {
            $configs[] = array(
                'client' => $clientName,
                'secret' => $this->astman->database_get($clientName, 'key') ?: null,
                'dids' => isset($clientDidsMap[$clientName]) ? $clientDidsMap[$clientName] : array(),
            );
        }

        return $configs;
    }

    /**
     * @return bool
     */
    private function isConfigured()
    {
        return !empty($this->astman->database_get('zruchnaio/calltracking', 'configured'))
            && !empty($this->getClientsConfiguration())
            && !empty($this->getDidsConfiguration())
        ;
    }

    /**
     * @deprecated use {@see self::getSequenceDatabaseValue()} method with "zruchnaio/calltracking/clients" argument instead
     *
     * @return string[]
     */
    private function getClientsConfiguration()
    {
        if ($clients = $this->getSequenceDatabaseValue('zruchnaio/calltracking/clients')) {
            return $clients;
        }

        // deprecated storage
        if ($clients = $this->astman->database_get('zruchnaio/calltracking', 'clients')) {
            \trigger_error('The "/zruchnaio/calltracking/clients" storage is deprecated, resave configuration to store it as "/zruchnaio/calltracking/clients/######"', \E_USER_DEPRECATED);

            return explode(',', $clients);
        }

        return [];
    }

    /**
     * @deprecated use {@see self::getSequenceDatabaseValue()} method with "zruchnaio/calltracking/dids" argument instead
     *
     * @return string[]
     */
    private function getDidsConfiguration()
    {
        if ($dids = $this->getSequenceDatabaseValue('zruchnaio/calltracking/dids')) {
            return $dids;
        }

        // deprecated storage
        if ($dids = $this->astman->database_get('zruchnaio/calltracking', 'dids')) {
            \trigger_error('The "/zruchnaio/calltracking/dids" storage is deprecated, resave configuration to store it as "/zruchnaio/calltracking/dids/######"', \E_USER_DEPRECATED);

            return explode(',', $dids);
        }

        return [];
    }

    /**
     * @param string $family
     *
     * @return string[]
     */
    protected function getSequenceDatabaseValue($family)
    {
        $values = $this->astman->database_show($family);

        if (!$values) {
            return [];
        }

        ksort($values, SORT_STRING);

        return array_values($values);
    }

    /**
     * @param string   $family
     * @param string[] $values
     *
     * @return void
     */
    protected function putSequenceDatabaseValue($family, array $values)
    {
        /** @var int<0, max> $idx */
        $idx = 0;

        foreach ($values as $value) {
            $key = sprintf('%06d', ++$idx);
            $this->astman->database_put($family, $key, $value);
        }
    }

    private function addAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $srcFilePath = __DIR__.'/agi/'.$filename.'.sh';
        $dstFilePath = $agibin_dest.'/'.$filename.'.sh';

        if (file_exists($dstFilePath)) {
            return;
        }

        symlink($srcFilePath, $dstFilePath);
        chmod($dstFilePath, 0755);
    }

    private function removeAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $dstFilePath = $agibin_dest.'/'.$filename.'.sh';

        if (file_exists($dstFilePath)) {
            unlink($dstFilePath);
        }
    }

    public function backup()
    {
        // TODO: Implement backup() method.
    }

    public function restore($backup)
    {
        // TODO: Implement restore() method.
    }

    /**
     * Hook into Dialplan (extensions_additional.conf)
     * @method doDialplanHook
     *
     * @param \extensions $ext The Extensions Class https://wiki.freepbx.org/pages/viewpage.action?pageId=98701336
     * @param string $engine Always Asterisk, Legacy
     * @param string $priority Priority
     */
    public function doDialplanHook($ext, $engine, $priority)
    {
        if (!$this->isConfigured()) {
            return;
        }

        $this->addCalltrackAnswerContext($ext);
        $this->addCalltrackNoanswerContext($ext);

        $this->modifyInCallEndContext($ext);
        $this->modifyMisscallContext($ext);
    }

    private function addCalltrackAnswerContext(\extensions $ext)
    {
        $section = 'calltrack-answer';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_execif('$["${DB(calltrack/${FROM_DID})}" != ""]', 'System', $this->getBinFileExec('calltracking').' "${CALLERID(num)}" "${DB(${FROM_DID}/${CALLERID(num)}/num)}" "${CHANNEL(LINKEDID)}" "${timecall}" "${FROM_DID}" "ANSWERED" "${DB(calltrack/${FROM_DID})}" "${DB(${DB(calltrack/${FROM_DID})}/key)}" "${CHANNEL(accountcode)}" &'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function addCalltrackNoanswerContext(\extensions $ext)
    {
        $section = 'calltrack-noanswer';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_execif('$["${DB(calltrack/${FROM_DID})}" != ""]', 'System', $this->getBinFileExec('calltracking').' "${CALLERID(num)}" "${DB(${FROM_DID}/${CALLERID(num)}/num)}" "${CHANNEL(LINKEDID)}" "${timecall}" "${FROM_DID}" "NO ANSWER" "${DB(calltrack/${FROM_DID})}" "${DB(${DB(calltrack/${FROM_DID})}/key)}" "${CHANNEL(accountcode)}" &'));
        $ext->add($section, $extension, '', new \ext_return());
    }

    private function modifyInCallEndContext(\extensions $ext)
    {
        $section = 'in-call-end';

        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }

        $extension = 's';

        // region Prepend custom commands before "dbdel" tag
        $ext->splice($section, $extension, 'dbdel', new \ext_execif('$["${innumber}" != ""]', 'Gosub', 'calltrack-answer,s,1'));
        // endregion Prepend custom commands before "dbdel" tag
    }

    private function modifyMisscallContext(\extensions $ext)
    {
        $section = 'misscall';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Required "%s" section not found', $section));
        }

        $extension = 's';

        // region Prepend custom commands before "dbdel" tag
        $ext->splice($section, $extension, 'dbdel', new \ext_gosub('calltrack-noanswer,s,1'));
        // endregion Prepend custom commands before "dbdel" tag
    }

    /**
     * @param string $binFileName
     * @param array<string, scalar> $environments
     * @return string
     */
    private function getBinFileExec($binFileName, array $environments = array())
    {
        $prefix = '';
        foreach ($environments as $name => $value) {
            $prefix .= sprintf('%s=\'%s\' ', $name, strtr($value, array('\'' => '\\\'')));
        }

        return $prefix.'/bin/bash '.$this->escapeExtData(escapeshellarg(__DIR__.'/bin/'.$binFileName.'.sh'));
    }

    private function escapeExtData($value)
    {
        return strtr($value, array(
            ';' => '\;',
            ')' => '\)',
        ));
    }

    public function doConfigPageInit($page)
    {
    }

    public function getActionBar($request)
    {
        $buttons = array();
        switch ($_GET['display']) {
            case 'calltracking':
                $buttons = array(
                    'delete' => array(
                        'name' => 'delete',
                        'id' => 'delete',
                        'value' => _('Delete')
                    ),
                    'reset' => array(
                        'name' => 'reset',
                        'id' => 'reset',
                        'value' => _('Reset')
                    ),
                    'submit' => array(
                        'name' => 'submit',
                        'id' => 'submit',
                        'value' => _('Submit')
                    )
                );
                if (empty($_GET['extdisplay'])) {
                    unset($buttons['delete']);
                }
                break;
        }
        return $buttons;
    }

    public function showPage()
    {
        $view = empty($_REQUEST['view']) ? 'main' : $_REQUEST['view'];

        if ('POST' === $_SERVER['REQUEST_METHOD']) {
            $_SERVER['REQUEST_URI'] = preg_replace(
                ['~error(?:=[^&#]*)?~', '~&{2,}~'],
                '&',
                $_SERVER['REQUEST_URI']
            );

            $simpleConfig = trim(preg_replace('~[\r\n]+~', "\n", $_POST['simple_config']));
            $configs = array();
            foreach (explode("\n", $simpleConfig) as $line) {
                list($did, $client, $secret) = explode(' ', trim(preg_replace('~\s+~', ' ', $line)));

                if (!isset($configs[$client])) {
                    $configs[$client] = array(
                        'client' => $client,
                        'secret' => $secret,
                        'dids' => array(),
                    );
                }

                $configs[$client]['dids'][] = $did;
            }

            $_SESSION['previous_value'] = $configs;

            try {
                $this->assertConfigs($configs);
                $this->eraseConfigs();
                $this->applyConfigs($configs);
            } catch (\InvalidArgumentException $e) {
                $_SERVER['REQUEST_URI'] .= false !== strpos($_SERVER['REQUEST_URI'], '?') ? '&' : '?';
                $_SERVER['REQUEST_URI'] .= 'error='.rawurlencode($e->getMessage());
            }

            \needreload();

            header('Location: '.$_SERVER['REQUEST_URI'], true, 302);
            return '';
        }

        $simpleConfig = '';
        foreach ($this->getConfigs() as $config) {
            foreach ($config['dids'] as $did) {
                $simpleConfig .= sprintf("%s %s %s\n", $did, $config['client'], $config['secret']);
            }
        }

        return load_view(__DIR__ . '/views/main.php', [
            // TODO
            'simpleConfig' => $simpleConfig,
            'previousValue' => $_SESSION['previous_value'],
            'error' => isset($_GET['error']) ? $_GET['error'] : '',
        ]);
    }

    /**
     * @param string $req
     * @param array $setting
     * @return bool
     *
     * @noinspection PhpUnused
     */
    public function ajaxRequest($req, &$setting)
    {
        switch ($req) {
            case 'ping':
                $this->allowAnonymous($setting);

                return true;
            default:
                return false;
        }
    }

    public function ajaxHandler()
    {
        switch ($_REQUEST['command']) {
            case 'ping':
                return 'pong';
            default:
                return false;
        }
    }

    public function getRightNav($request)
    {
        return <<<'HTML'
<div class="bootstrap-table">
  <div class="fixed-table-toolbar">
    <div class="bs-bars">
      <div class="toolbar-cbbnav">
        <a href="?display=calltracking" class="btn btn-default">Main</a>
      </div>
    </div>
  </div>
</div>
HTML;
    }

    /**
     * @return void
     */
    private function allowAnonymous(array &$setting)
    {
        $setting['allowremote'] = true;
        $setting['authenticate'] = false;
    }
}
