<?php

namespace Zruchna\FreePbx\Autorecaller\Dialplan;

use Symfony\Component\Console\Command\Command;

class ZruchnaAutorecallerAgiExtension extends \ext_agi implements DialplanCommandInterface
{
    /** @var bool|null */
    protected static $prependFastAGI = null;

    /** @var Command */
    private $command;

    /** @var array<string, scalar> */
    private $arguments;

    /** @var array<string, scalar|scalar[]> */
    private $options;

    /**
     * @param string $commandClass
     * @param string[] $arguments
     */
    public function __construct($commandClass, array $arguments = array(), array $options = array())
    {
        if (!class_exists($commandClass)) {
            throw new \InvalidArgumentException(sprintf('Unable to find "%s" command class', $commandClass));
        }

        $command = new $commandClass();
        if (!$command instanceof Command) {
            throw new \InvalidArgumentException(sprintf('Expected to command class be extended %s class, %s provided', Command::class, $commandClass));
        }

        $this->assertArguments($command, $arguments);
        $this->assertOptions($command, $options);

        $this->command = $command;
        $this->arguments = $arguments;
        $this->options = $options;

        parent::__construct($this->getApplicationData());
    }

    public function getApplication()
    {
        return 'AGI';
    }

    public function getData()
    {
        return $this->getApplicationData();
    }

    /**
     * @return string
     */
    private function getApplicationData()
    {
        $data = array('zruchna-autorecaller', $this->command->getName());

        foreach ($this->options as $optionName => $optionValue) {
            if (\is_array($optionValue)) {
                foreach ($optionValue as $optionValueItem) {
                    $data[] = $this->normalizeOption($optionName, (string) $optionValueItem);
                }
            } elseif (\is_bool($optionValue)) {
                if ($optionValue) {
                    $data[] = $this->normalizeOption($optionName, null);
                }
            } else {
                $data[] = $this->normalizeOption($optionName, (string) $optionValue);
            }
        }

        foreach ($this->command->getDefinition()->getArguments() as $commandArgument) {
            $argumentName = $commandArgument->getName();

            if (isset($this->arguments[$argumentName])) {
                $data[] = $this->arguments[$argumentName];
            }
        }

        return implode(',', $data);
    }

    /**
     * @throws \InvalidArgumentException
     * @return void
     */
    protected function assertArguments(Command $command, array $providedArguments)
    {
        $commandName = $command->getName();
        $commandDefinition = $command->getDefinition();

        $commandArguments = array();
        foreach ($commandDefinition->getArguments() as $argument) {
            $commandArguments[$argument->getName()] = $argument->isRequired();
        }

        if (array() !== $extraArguments = array_diff_key($providedArguments, $commandArguments)) {
            throw new \InvalidArgumentException(sprintf('Extra arguments provided for "%s" command: %s', $commandName, implode(', ', array_keys($extraArguments))));
        }
        if (array() !== $missedArguments = array_diff_key(array_filter($commandArguments), $providedArguments)) {
            throw new \InvalidArgumentException(sprintf('Required arguments missed for "%s" command: %s', $commandName, implode(', ', $missedArguments)));
        }
    }

    /**
     * @throws \InvalidArgumentException
     * @return void
     */
    protected function assertOptions(Command $command, array $providedOptions)
    {
        $commandName = $command->getName();
        $commandDefinition = $command->getDefinition();

        $commandOptions = array();
        foreach ($commandDefinition->getOptions() as $option) {
            $commandOptions[$option->getName()] = true;
        }

        if (array() !== $extraOptions = array_diff_key($providedOptions, $commandOptions)) {
            throw new \InvalidArgumentException(sprintf('Extra options provided for "%s" command: %s', $commandName, implode(', ', array_keys($extraOptions))));
        }
    }

    /**
     * @param string $optionName
     * @param string|null $optionValue
     * @return string
     */
    protected function normalizeOption($optionName, $optionValue)
    {
        if (null === $optionValue) {
            return sprintf('--%s', $optionName);
        }

        return sprintf('--%s=%s', $optionName, $optionValue);
    }
}
