<?php

namespace Zruchna\FreePbx\Autorecaller\Dialplan;

use Zruchna\FreePbx\Autorecaller\Command\CancelRecallCommand;
use Zruchna\FreePbx\Autorecaller\Command\RegisterMissedCallCommand;
use Zruchna\FreePbx\Autorecaller\Command\RegisterSuccessfulRecallCommand;
use Zruchna\FreePbx\Autorecaller\Command\RegisterUnsuccessfulAttemptCommand;
use Zruchna\FreePbx\Autorecaller\Command\RescheduleRecallCommand;
use Zruchna\FreePbx\Autorecaller\Entity\AutorecallerRule;
use Zruchna\FreePbx\Autorecaller\Repository\AutorecallerRuleRepository;

class DialplanBuilder
{
    /** @var AutorecallerRuleRepository */
    private $autorecallerRuleRepository;

    public function __construct(AutorecallerRuleRepository $autorecallerRuleRepository)
    {
        $this->autorecallerRuleRepository = $autorecallerRuleRepository;
    }

    public function populateExtensions(\extensions $ext)
    {
        $this->addMissPlaybackContext($ext);
        $this->addMissdialoperatorsContext($ext);
        $this->addMisscallavtoContext($ext);
        $this->addNoworkContext($ext);

        $this->modifyMisscallContext($ext);
        $this->modifyMacroOutdialContext($ext);
        $this->modifyCallanswerContext($ext, 'callanswer');
        $this->modifyCallanswerContext($ext, 'callanswe');
    }

    /**
     * @return AutorecallerRule[] Detached entities.
     */
    private function getAutorecallerRules()
    {
        $rules = array();

        foreach ($this->autorecallerRuleRepository->findAll() as $rule) {
            $recaller = $rule->getAutorecaller();

            if (!$recaller->isEnabled()) {
                continue;
            }

            $recording = $rule->getRecording() ?: $recaller->getRecording();

            $rule = clone $rule;
            $rule->setRecording($recording);

            $rules[] = $rule;
        }

        return $rules;
    }

    private function addMissPlaybackContext(\extensions $ext)
    {
        $section = 'miss-playback';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_answer());

        // Проигрывание мелодии.
        foreach ($this->getAutorecallerRules() as $rule) {
            $extension = sprintf('%d', $rule->getId());
            $recordingFilename = $rule->getRecording()->getFilename();

            $ext->add($section, $extension, '', new \ext_playback($recordingFilename.',noanswer'));

            $ext->add($section, $extension, '', new \ext_return());
        }

        $extension = 'h';
        $ext->add($section, $extension, '', new \ext_set('__attempt', '${MATH(${attempt}+1,int)}')); // TODO: Is this need?
        $ext->add($section, $extension, '', new \ext_goto('1', 's', 'misscall'));
    }

    private function addMissdialoperatorsContext(\extensions $ext)
    {
        $section = 'missdialoperators';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_set('__clientnum', '${CALLERID(num)}'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(number)', '${__clientnum}'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(all)', '${__clientnum}'));
        $ext->add($section, $extension, '', new \ext_set('CALLERID(name)', '${__clientnum}'));

        // Запускается проверка рабочего времени.
        // TODO: Это должно генерироваться исходя из настроек.
        //
        // У успешном случае timeconditions должен направить звонок туда, куда это было настроено.
        // Далее это будет работать схожим со входящей маршрутизацией образом.
        //
        // В неуспешном случае timeconditions должен направлять канал в nowork,s,1.

        foreach ($this->getAutorecallerRules() as $rule) {
            $extension = sprintf('%d', $rule->getId());

            $timeconditionId = $rule->getTimeconditionId() ?: $rule->getAutorecaller()->getTimeconditionId();

            $ext->add($section, $extension, '', new \ext_goto('1', sprintf('%d', $timeconditionId), 'timeconditions'));
        }
    }

    private function addMisscallavtoContext(\extensions $ext)
    {
        $section = 'misscallavto';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_set('__ZRUCHNA_UNSUCCESSFUL_RECALL_REASON', 'agent'));
        $ext->add($section, $extension, '', new \ext_set('__klientnum', '${abon}'));
        $ext->add($section, $extension, '', new \ext_set('__attempt', '${atmp}'));

        $extension = '_.';
        $ext->add($section, $extension, '', new \ext_gosub('1', '${EXTEN}', 'miss-playback'));
        $ext->add($section, $extension, '', new \ext_dial('local/${abon}@from-internal', '300,tT'));

        $extension = 'h';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_set('__attempt', '${MATH(${attempt}+1,int)}'));
        $ext->add($section, $extension, '', new \ext_noop('zruchna.io Autorecaller: Current dialstatus: "${DIALSTATUS}"'));

        $ext->add($section, $extension, '', new \ext_set('__ZRUCHNA_UNSUCCESSFUL_RECALL_REASON', ''));
        $ext->add($section, $extension, '', new \ext_execif('$["${DIALSTATUS}" = "CANCEL" | "${DIALSTATUS}" = "CONGESTION" | "${DIALSTATUS}" = ""]', 'Set', '__ZRUCHNA_UNSUCCESSFUL_RECALL_REASON=agent'));
        $ext->add($section, $extension, '', new \ext_execif('$["${DIALSTATUS}" = "BUSY"]', 'Set', '__ZRUCHNA_UNSUCCESSFUL_RECALL_REASON=caller'));
        $ext->add($section, $extension, '', new \ext_execif('$["${DIALSTATUS}" = "CHANUNAVAIL" | "${DIALSTATUS}" = "FAILED"]', 'Set', '__ZRUCHNA_UNSUCCESSFUL_RECALL_REASON=error'));

        $ext->add($section, $extension, '', new \ext_noop('zruchna.io Autorecaller: Unsuccessfull recall reason: ${ZRUCHNA_UNSUCCESSFUL_RECALL_REASON}'));

        $ext->add($section, $extension, '', new \ext_execif('$["${DIALSTATUS}" != "ANSWER" & "${DIALSTATUS}" != "ANSWERED"]', 'Set', 'CALLERID(num)=${klientnum}'));
        $ext->add($section, $extension, '', new \ext_gotoif('$["${DIALSTATUS}" != "ANSWER" & "${DIALSTATUS}" != "ANSWERED"]', 'misscall,s,1'));

        $ext->add($section, $extension, '', new ZruchnaAutorecallerAgiExtension(
            RegisterSuccessfulRecallCommand::class,
            array(
                'recall-id' => '${ZRUCHNA_RECALL_ID}',
            ),
            array(
                'call-id' => '${CHANNEL(accountcode)}',
                'caller-id' => '${abon}',
            )
        ));

        $ext->add($section, $extension, '', new \ext_macro('hangupcall'));
    }

    private function addNoworkContext(\extensions $ext)
    {
        $section = 'nowork';
        $ext->addSectionNoCustom($section, true);

        $extension = 's';
        $ext->add($section, $extension, '', new \ext_noop());
        $ext->add($section, $extension, '', new \ext_system('echo "${CALLERID(num)}" >> /var/log/asterisk/nowork-miss.txt'));

        $ext->add($section, $extension, '', new ZruchnaAutorecallerAgiExtension(
            RescheduleRecallCommand::class,
            array(
                'recall-id' => '${ZRUCHNA_RECALL_ID}',
            ),
            array(
                'caller-id' => '${CALLERID(num)}',
                'call-id' => '${CDR(accountcode)}',
            )
        ));

        $ext->add($section, $extension, '', new \ext_hangup());
    }

    private function modifyMisscallContext(\extensions $ext)
    {
        $section = 'misscall';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException('Unable to find required "misscall" section');
        }

        $extension = 's';

        // region Appending lines after first command
        $afterPriority = 1;

        // Регистрация пропущенного вызова
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "ATS"]', 'System', '/var/scripts/callback1.sh "${CALLERID(num)}" "${CDR(accountcode)}" "${STRFTIME(${EPOCH},,%Y-%m-%d %H:%M:%S)}" "${istochnik}" &'));

        // Вызов команды на регистрацию пропущенного звонка
        $ext->splice($section, $extension, $afterPriority++, new ExecIfExtension(
            '$["${CHANNEL(accountcode):0:3}" = "ATS"]',
            new ZruchnaAutorecallerAgiExtension(
                RegisterMissedCallCommand::class,
                array(
                    'caller-id' => '${CALLERID(num)}',
                ),
                array(
                    'call-id' => '${CDR(accountcode)}',
                    'from-did' => '${FROM_DID}',
                )
            )
        ));

        // Регистрация неудачного перезвона
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "MS1"]', 'System', '/var/scripts/callback2.sh "${CALLERID(num)}" "${CDR(accountcode)}" "${STRFTIME(${EPOCH},,%Y-%m-%d %H:%M:%S)}" "${istochnik}" &'));
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:2}" = "NW"]', 'System', '/var/scripts/callback2.sh "${CALLERID(num)}" "${CDR(accountcode)}" "${STRFTIME(${EPOCH},,%Y-%m-%d %H:%M:%S)}" "${istochnik}" &'));
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "MS2"]', 'System', '/var/scripts/callback3.sh "${CALLERID(num)}" "${CDR(accountcode)}" "${STRFTIME(${EPOCH},,%Y-%m-%d %H:%M:%S)}" "${istochnik}" &'));

        // Вызов команды на регистрацию неуспешного перезвона
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('${CHANNEL(accountcode)}'));

        $ext->splice($section, $extension, $afterPriority++, new ExecIfExtension(
            '$["${CHANNEL(accountcode):0:2}" = "MS"]',
            new ZruchnaAutorecallerAgiExtension(
                RegisterUnsuccessfulAttemptCommand::class,
                array(
                    'recall-id' => '${ZRUCHNA_RECALL_ID}',
                ),
                array(
                    'caller-id' => '${CALLERID(num)}',
                    'call-id' => '${CDR(accountcode)}',
                )
            )
        ));
        // endregion Appending lines after first command

        // region Replace "userfieldmod" tag
        $replacePriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as $priority => $command) {
            if ('userfieldmod' === $command['tag']) {
                $replacePriority = $priority + 1;
            }
        }
        if (null === $replacePriority) {
            throw new \UnexpectedValueException('Unable to find "userfieldmod" tag on "misscall" context');
        }

        // Установка значения "miss_call" в поле "userfield" в таблице CDR.
        $ext->replace($section, $extension, $replacePriority, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "ATS"]', 'Set', 'CDR(userfield)=miss_call'));
        $ext->replace($section, $extension, $replacePriority, new \ext_execif('$["${CHANNEL(accountcode):0:2}" = "MS"]', 'Set', 'CDR(userfield)=recall_fail'));

        $afterPriority = $replacePriority;

        // TODO: Пересмотреть значения и код ниже для отметки неуспешных перезвонов.
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:2}" = "NW"]', 'Set', 'CDR(userfield)=miss_call'));
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "MS1"]', 'Set', 'CDR(userfield)=miss_call_1'));
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "MS2"]', 'Set', 'CDR(userfield)=miss_call_2'));
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${CHANNEL(accountcode):0:3}" = "MS3"]', 'Set', 'CDR(userfield)=miss_call_3'));

        // endregion Replace "userfieldmod" tag
    }

    private function modifyMacroOutdialContext(\extensions $ext)
    {
        $section = 'macro-outdial';
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException('Unable to find required "macro-outdial" section');
        }

        $extension = 's';

        // region Appending lines after first command
        $afterPriority = 1;

        // Номер звонящего пишется в переменную todialklinetnum для дельнейшего использования.
        $ext->splice($section, $extension, $afterPriority++, new \ext_set('__todialklinetnum', '${CALLERID(num)}'));

        // Логируются префиксы accountcode.
        $ext->splice($section, $extension, $afterPriority++, new \ext_noop('"${CALLERID(num):0:3}" "${CALLERID(num):0:2}"'));

        // Нормализуется номер телефона звонящего.
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${todialklinetnum:0:3}" = "375"]', 'Set', '__todialklinetnum=+375${todialklinetnum:3}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${todialklinetnum:0:2}" = "00"]', 'Set', '__todialklinetnum=+${todialklinetnum:2}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${todialklinetnum:0:2}" = "80"]', 'Set', '__todialklinetnum=+375${todialklinetnum:2}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${todialklinetnum:0:1}" = "7"]', 'Set', '__todialklinetnum=+{todialklinetnum}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$[${LEN(${todialklinetnum})} = 6]', 'Set', '__todialklinetnum=+375212${todialklinetnum}'));

        // Запуск комманды на отмену перезвона.
        $ext->splice($section, $extension, $afterPriority++, new ZruchnaAutorecallerAgiExtension(
            CancelRecallCommand::class,
            array(
                'caller-id' => '${todialklinetnum}',
            ),
            array(
                'reason' => 'outbound-call-answer',
                'call-id' => '${CHANNEL(accountcode)}',
            )
        ));

        // endregion Appending lines after first command
    }

    private function modifyCallanswerContext(\extensions $ext, $section)
    {
        if (!$ext->section_exists($section)) {
            throw new \UnexpectedValueException(sprintf('Unable to find required "%s" section', $section));
        }

        $extension = 's';

        // region Appending lines after "giveaccountcode" tag
        $afterPriority = null;
        foreach ($ext->_exts[$section][' '.$extension.' '] as  $priority => $command) {
            if ('giveaccountcode' === $command['tag']) {
                $afterPriority = $priority + 1;
            }
        }
        if (null === $afterPriority) {
            throw new \UnexpectedValueException(sprintf('Unable to find "giveaccountcode" tag on "%s" section', $section));
        }

        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${clientnum}" != "" & "${CHANNEL(accountcode):0:2}" != "MS" ]', 'AGI', 'zruchna-autorecaller,touch-misssed-call-to-recall,${clientnum},${CALLERID(num)}'));
        // $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${clientnum}" != "" & "${CHANNEL(accountcode):0:2}" != "NW" ]', 'AGI', 'zruchna-autorecaller,touch-misssed-call-to-recall,${clientnum},${CALLERID(num)}'));
        $ext->splice($section, $extension, $afterPriority++, new \ext_execif('$["${clientnum}" != "" & "${CHANNEL(accountcode):0:2}" = "ATS" ]', 'System', '/var/scripts/insert2.sh "${STRFTIME(${EPOCH},,%Y-%m-%d %H:%M:%S)}" "${clientnum}" "${CALLERID(num)}" &'));

        // Вызов комманды на отмену перезвона.
        $ext->splice($section, $extension, $afterPriority++, new ExecIfExtension(
            '$["${clientnum}" != "" & "${CHANNEL(accountcode):0:3}" = "ATS" ]',
            new ZruchnaAutorecallerAgiExtension(
                CancelRecallCommand::class,
                array(
                    'caller-id' => '${CALLERID(num)}',
                ),
                array(
                    'reason' => 'inbound-call-answer',
                    'call-id' => '${CHANNEL(accountcode)}',
                )
            )
        ));
        // endregion Appending lines after first command
    }
}
