<?php

namespace Zruchna\FreePbx\Autorecaller\Command;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DBALException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\StyleInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Zruchna\FreePbx\Autorecaller\Clock\ClockInterface;
use Zruchna\FreePbx\Autorecaller\Doctrine\DBAL\Type\DatetimeImmutableUtcType;
use Zruchna\FreePbx\Autorecaller\Factory\ClockFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Doctrine\ConnectionFactory;

class VacuumScheduledCallsTableCommand extends Command
{
    protected static $defaultName = 'vacuum-scheduled-calls-table';

    /** @var (OutputInterface|StyleInterface)|null */
    private $io = null;

    /** @var Connection|null */
    private $connection = null;

    /** @var ClockInterface|null */
    private $clock = null;

    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->io) {
            $this->io = new SymfonyStyle($input, $output);
        }
        if (null === $this->connection) {
            $this->connection = ConnectionFactory::getConnection();
        }
        if (null === $this->clock) {
            $this->clock = ClockFactory::getClock();
        }
    }

    protected function configure()
    {
        $this->addOption('interval', null, InputOption::VALUE_REQUIRED, 'In days.', 14);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $interval = $input->getOption('interval');

        if (!\is_int($interval) && !\is_numeric($interval)) {
            $this->io->error('Interval must be an integer value');

            return 2;
        }

        if ($interval < 1) {
            $this->io->error('Too small interval provided. Interval must be equal or greater then 1.');

            return 1;
        }

        try {
            $interval = new \DateInterval(sprintf('P%dD', $interval));
        } catch (\Exception $e) {
            $this->io->error('Unable to create date interval: '.$e->getMessage());

            return 1;
        }

        $edgeDate = $this->clock->now()
            ->sub($interval)
        ;

        $query = <<<'SQL'
DELETE
FROM zruchna_autorecaller_scheduled_recalls
WHERE calldate <= :edge_date
SQL;

        try {
            $stmt = $this->connection->prepare($query);
        } catch (DBALException $e) {
            $this->io->error('Unable to prepare SQL query: '.$e->getMessage());

            return 1;
        }

        $stmt->bindValue('edge_date', $edgeDate, DatetimeImmutableUtcType::DATETIME_IMMUTABLE_UTC);

        try {
            $stmt->execute();
        } catch (DBALException $e) {
            $this->io->error('Unable to execute SQL query: '.$e->getMessage());

            return 1;
        }

        if (0 === $affectedRowsCount = $stmt->rowCount()) {
            $this->io->success('Unable to find scheduled records older then provided interval');
        } else {
            $this->io->success(sprintf('Successfully removed %d scheduled records', $affectedRowsCount));
        }

        return 0;
    }
}
