<?php

namespace Zruchna\FreePbx\Autorecaller\Command;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Zruchna\FreePbx\Autorecaller\Entity\Autorecaller;
use Zruchna\FreePbx\Autorecaller\Entity\ScheduledRecall;
use Zruchna\FreePbx\Autorecaller\Factory\Doctrine\EntityManagerFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Repository\SimpleRepositoryFactory;
use Zruchna\FreePbx\Autorecaller\Repository\ScheduledRecallRepository;

class RescheduleRecallCommand extends Command
{
    protected static $defaultName = 'reschedule-recall';

    /** @var EntityManagerInterface|null */
    private $entityManager = null;

    /** @var ScheduledRecallRepository|null */
    private $scheduledRecallRepository = null;

    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->entityManager) {
            $this->entityManager = EntityManagerFactory::getEntityManager();
        }
        if (null === $this->scheduledRecallRepository) {
            $this->scheduledRecallRepository = SimpleRepositoryFactory::getRepository(ScheduledRecall::class);
        }
    }

    protected function configure()
    {
        $this->setProcessTitle('zruchna.io · Autorecaller');

        $this->addArgument('recall-id', InputArgument::REQUIRED);

        $this->addOption('call-id', null, InputOption::VALUE_OPTIONAL);
        $this->addOption('caller-id', null, InputOption::VALUE_OPTIONAL);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $recallId = $input->getArgument('recall-id');
        // $callId = $input->getOption('call-id');
        // $callerId = $input->getOption('caller-id');

        $io = new SymfonyStyle($input, $output);

        if (!\is_int($recallId) && !\is_numeric($recallId)) {
            if (!$output->isQuiet()) {
                $io->error('"recall-id" argument must be an integer value.');
            }

            return 2;
        }

        $recall = $this->scheduledRecallRepository->find($recallId);
        if (null === $recall) {
            if (!$output->isQuiet()) {
                $io->error('Provided recall not found.');
            }

            return 1;
        }

        $rule = $recall->getAutorecallerRule();
        $recaller = $rule->getAutorecaller();

        $nextCalldate = $recall->getCalldate()
            ->modify('tomorrow midnight')
        ;

        $nextCalldateDelay = $recaller->getWorktimeStartsFrom();

        $output->writeln('$nextCalldateDelay = '.$nextCalldateDelay);

        $countOfNextWorkdayRecalls = $this->getNextWorkdayRecallsCount($recaller, $nextCalldate);
        $nextCalldateDelay += $recaller->getWorktimeStartInterval() * $countOfNextWorkdayRecalls;

        $output->writeln('$nextCalldateDelay = '.$nextCalldateDelay);

        $nextCalldateInterval = new \DateInterval(sprintf('PT%dS', $nextCalldateDelay));
        $nextCalldate = $nextCalldate->add($nextCalldateInterval);

        $output->writeln('$nextCalldate = '.$nextCalldate->format('Y-m-d H:i:s'));

        $recall
            ->setCalldate($nextCalldate)
            ->setCallId(null)
        ;

        $this->entityManager->flush();

        return 0;
    }

    private function getNextWorkdayRecallsCount(Autorecaller $recaller, \DateTimeImmutable $nextWorkday)
    {
        return $this->scheduledRecallRepository->countOfDay($nextWorkday, $recaller);
    }
}
