<?php

namespace Zruchna\FreePbx\Autorecaller\Command;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Zruchna\FreePbx\Autorecaller\Builder\ScheduledRecallBuilder;
use Zruchna\FreePbx\Autorecaller\Entity\Autorecaller;
use Zruchna\FreePbx\Autorecaller\Entity\AutorecallerRule;
use Zruchna\FreePbx\Autorecaller\Factory\Builder\ScheduledRecallBuilderFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Doctrine\EntityManagerFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Repository\SimpleRepositoryFactory;
use Zruchna\FreePbx\Autorecaller\Repository\AutorecallerRepository;
use Zruchna\FreePbx\Autorecaller\Repository\AutorecallerRuleRepository;

class RegisterMissedCallCommand extends Command
{
    protected static $defaultName = 'register-missed-call';

    /** @var EntityManagerInterface|null */
    private $entityManager = null;

    /** @var AutorecallerRepository|null */
    private $autorecallerRepository = null;

    /** @var AutorecallerRuleRepository|null */
    private $autorecallerRuleRepository = null;

    /** @var ScheduledRecallBuilder|null */
    private $scheduledRecallBuilder = null;

    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->entityManager) {
            $this->entityManager = EntityManagerFactory::getEntityManager();
        }
        if (null === $this->autorecallerRepository) {
            $this->autorecallerRepository = SimpleRepositoryFactory::getRepository(Autorecaller::class);
        }
        if (null === $this->autorecallerRuleRepository) {
            $this->autorecallerRuleRepository = SimpleRepositoryFactory::getRepository(AutorecallerRule::class);
        }
        if (null === $this->scheduledRecallBuilder) {
            $this->scheduledRecallBuilder = ScheduledRecallBuilderFactory::getScheduledRecallBuilder();
        }
    }

    protected function configure()
    {
        $this->setProcessTitle('zruchna.io · Autorecaller');

        $this->addArgument('caller-id', InputArgument::REQUIRED);

        $this->addOption('use-recaller', null, InputOption::VALUE_OPTIONAL);
        $this->addOption('call-id', null, InputOption::VALUE_OPTIONAL);
        $this->addOption('from-did', null, InputOption::VALUE_REQUIRED);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $callerId = $input->getArgument('caller-id');
        $useRecaller = $input->getOption('use-recaller');
        // $callId = $input->getOption('call-id');
        $fromDid = $input->getOption('from-did');

        if (\is_int($useRecaller) || \is_numeric($useRecaller)) {
            $concreteRecaller = $this->findRecaller((int) $useRecaller);
        } else {
            $concreteRecaller = null;
        }

        if (null !== $concreteRecaller) {
            $targetRecallers = array($concreteRecaller);
        } else {
            $targetRecallers = $this->findEnabledRecallers($fromDid);
        }

        $targetRules = array();
        foreach ($targetRecallers as $recaller) {
            if (null !== $targetRule = $this->findNextRule($recaller)) {
                $targetRules[] = $targetRule;
            }
        }

        $this->scheduledRecallBuilder
            ->reset()
            ->setCallerId($callerId)
        ;

        foreach ($targetRules as $rule) {
            $scheduledRecall = $this->scheduledRecallBuilder
                ->setRule($rule)
                ->getScheduledRecall();

            $this->entityManager->persist($scheduledRecall);
        }

        $this->entityManager->flush();
    }

    /**
     * @param int $recallerId
     * @return Autorecaller|null
     */
    private function findRecaller($recallerId)
    {
        return $this->autorecallerRepository->find($recallerId);
    }

    /**
     * @return Autorecaller[]
     */
    private function findEnabledRecallers($fromDid)
    {
        return array_merge(
            $this->autorecallerRepository->findByDid($fromDid, true),
            $this->autorecallerRepository->findWithoutDid(true)
        );
    }

    /**
     * @return AutorecallerRule|null
     */
    private function findNextRule(Autorecaller $recaller)
    {
        return $this->autorecallerRuleRepository->findNextRule($recaller, 0);
    }
}
