<?php

namespace Zruchna\FreePbx\Autorecaller\Repository;

use Doctrine\DBAL\Types\Type;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\NonUniqueResultException;
use Doctrine\ORM\NoResultException;
use Zruchna\FreePbx\Autorecaller\Entity\Autorecaller;
use Zruchna\FreePbx\Autorecaller\Entity\ScheduledRecall;

/**
 * @method ScheduledRecall|null find($id, $lockMode = null, $lockVersion = null)
 * @method ScheduledRecall[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 * @method ScheduledRecall|null findOneBy(array $criteria, array $orderBy = null)
 * @method ScheduledRecall[]    findAll()
 */
class ScheduledRecallRepository extends AbstractSimpleRepository
{
    /** @var \DateTimeZone|null */
    private static $utc = null;

    public function __construct(EntityManagerInterface $em)
    {
        parent::__construct($em, ScheduledRecall::class);
    }

    /**
     * @param \DateTimeImmutable $date
     * @return ScheduledRecall[]
     */
    public function findActual(\DateTimeImmutable $date)
    {
        $rangeStart = $date->modify('30 minutes ago');
        $rangeEnd = $date;

        return $this->createQueryBuilder('r')
            ->andWhere('r.callId IS NULL')
            ->andWhere('r.calldate BETWEEN :rangeStart AND :rangeEnd')
            ->setParameter('rangeStart', $rangeStart, Type::DATETIME)
            ->setParameter('rangeEnd', $rangeEnd, Type::DATETIME)
            ->getQuery()
            ->getResult();
    }

    /**
     * @param string $callerId
     * @param bool $actualOnly
     * @return ScheduledRecall[]
     */
    public function findByCallerId($callerId, $actualOnly = false)
    {
        $qb = $this->createQueryBuilder('r')
            ->andWhere('r.callerId = :callerId')
            ->setParameter('callerId', $callerId, Type::STRING)
            ->orderBy('r.calldate', 'ASC')
        ;

        if ($actualOnly) {
            $qb->andWhere('r.callId IS NULL');
        }

        return $qb->getQuery()->getResult();
    }

    /**
     * @param Autorecaller|null $autorecaller
     * @return int
     */
    public function countOfDay(\DateTimeImmutable $date, $autorecaller = null)
    {
        $rangeStart = clone $date;
        $rangeStart
            ->setTimezone(self::utc())
            ->modify('midnight')
        ;

        $rangeEnd = clone $date;
        $rangeEnd
            ->setTimezone(self::utc())
            ->modify('tomorrow midnight -1 second')
        ;

        $qb = $this->createQueryBuilder('sr')
            ->select('COUNT(0)')
            ->andWhere('sr.calldate BETWEEN :rangeStart AND :rangeEnd')
            ->setParameter('rangeStart', $rangeStart, Type::DATETIME)
            ->setParameter('rangeEnd', $rangeEnd, Type::DATETIME)
        ;

        if (null !== $autorecaller) {
            $qb
                ->join('sr.autorecallerRule', 'ar')
                ->andWhere('ar.autorecaller = :autorecaller')
                ->setParameter('autorecaller', $autorecaller)
            ;
        }

        $query = $qb->getQuery();

        try {
            return $query->getSingleScalarResult();
        } catch (NoResultException $e) {
            return 0;
        } catch (NonUniqueResultException $e) {
            throw new \UnexpectedValueException('Must be only single result', 0, $e);
        }
    }

    /**
     * @return \DateTimeZone
     */
    private static function utc()
    {
        if (null === self::$utc) {
            self::$utc = new \DateTimeZone('UTC');
        }

        return self::$utc;
    }
}
