<?php

namespace Zruchna\FreePbx\Autorecaller\Factory;

use Composer\Autoload\ClassLoader;
use Doctrine\Common\Annotations\AnnotationRegistry;

abstract class AutoloaderFactory
{
    /** @var ClassLoader|null */
    private static $loader = null;

    /**
     * @param string $env
     * @return ClassLoader
     */
    public static function getLoader($env = 'prod')
    {
        if (null !== self::$loader) {
            return self::$loader;
        }

        self::$loader = $loader = new ClassLoader();

        foreach (self::getNamespaces() as $prefix => $paths) {
            $loader->add($prefix, $paths);
            $loader->addPsr4($prefix, $paths);
        }

        $loader->addClassMap(self::getClassMap());

        $loader->setClassMapAuthoritative('prod' === $env);

        $loader->register(true);

        AnnotationRegistry::registerLoader(array($loader, 'loadClass'));

        return $loader;
    }

    public static function dumpClassMap()
    {
        $baseDirPathLength = \strlen(__DIR__);

        $dumpFilename = self::getClassMapFilePath();

        $dumpFile = fopen($dumpFilename, 'wt');
        fwrite($dumpFile, "<?php\n\nreturn array(\n");

        foreach (self::iterateClassFiles(__DIR__, $baseDirPathLength) as $class => $relatedFilePath) {
            if (self::class === $class) {
                continue;
            }

            fprintf($dumpFile, "    %s => __DIR__.'/../../src'.%s,\n", var_export($class, true), var_export($relatedFilePath, true));
        }

        fwrite($dumpFile, ");\n");
        fclose($dumpFile);
    }

    private static function iterateClassFiles($directory, $baseDirPathLength)
    {
        $directoryIterator = new \RecursiveDirectoryIterator(
            $directory,
            \FilesystemIterator::NEW_CURRENT_AND_KEY
            | \FilesystemIterator::SKIP_DOTS
        );

        foreach ($directoryIterator as $fileInfo) {
            if ($fileInfo->isDir()) {
                foreach (self::iterateClassFiles($fileInfo->getPathname(), $baseDirPathLength) as $class => $relatedFilePath) {
                    yield $class => $relatedFilePath;
                }
            }

            if (!$fileInfo->isFile() || !$fileInfo->isReadable()) {
                continue;
            }
            if ('php' !== $fileInfo->getExtension()) {
                continue;
            }
            if ($fileInfo->getFilename() !== ucfirst($fileInfo->getFilename())) {
                continue;
            }

            $className = $fileInfo->getBasename('.php');
            $classNamespace = strtr(substr($fileInfo->getPath(), $baseDirPathLength + 1), DIRECTORY_SEPARATOR, '\\');
            $class = sprintf('Zruchna\FreePbx\Autorecaller\%s', ltrim(sprintf('%s\%s', $classNamespace, $className), '\\'));

            $relatedFilePath = substr($fileInfo->getPathname(), $baseDirPathLength);

            yield $class => $relatedFilePath;
        }
    }

    /**
     * @return array<string, string>
     */
    protected static function getClassMap()
    {
        if (file_exists(self::getClassMapFilePath())) {
            return require self::getClassMapFilePath();
        }

        return array();
    }

    /**
     * @return string
     */
    protected static function getClassMapFilePath()
    {
        return __DIR__.'/../../resources/autoloader'.'/class_map.php';
    }

    protected static function getNamespaces()
    {
        $basePath = __DIR__;
        $modulePath = basename(basename($basePath));
        $freePbxPath = basename(basename($modulePath));
        $composerLibsPath = $freePbxPath.'/libraries/Composer/vendor';

        return array(
            __NAMESPACE__.'\\' => $basePath,
            'Doctrine\ORM\\' => $composerLibsPath.'/doctrine/orm/lib/Doctrine/ORM/'
        );
    }
}
