<?php

namespace Zruchna\FreePbx\Autorecaller\Command;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Zruchna\FreePbx\Autorecaller\Builder\ScheduledRecallBuilder;
use Zruchna\FreePbx\Autorecaller\Entity\AutorecallerRule;
use Zruchna\FreePbx\Autorecaller\Entity\ScheduledRecall;
use Zruchna\FreePbx\Autorecaller\Factory\Builder\ScheduledRecallBuilderFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Doctrine\EntityManagerFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Repository\SimpleRepositoryFactory;
use Zruchna\FreePbx\Autorecaller\Repository\AutorecallerRuleRepository;
use Zruchna\FreePbx\Autorecaller\Repository\ScheduledRecallRepository;

class RegisterUnsuccessfulAttemptCommand extends Command
{
    protected static $defaultName = 'register-unsuccessful-attempt';

    /** @var EntityManagerInterface|null */
    private $entityManager = null;

    /** @var AutorecallerRuleRepository|null */
    private $autorecallerRuleRepository = null;

    /** @var ScheduledRecallRepository|null */
    private $scheduledRecallRepository = null;

    /** @var ScheduledRecallBuilder|null */
    private $scheduledRecallBuilder = null;

    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->entityManager) {
            $this->entityManager = EntityManagerFactory::getEntityManager();
        }
        if (null === $this->autorecallerRuleRepository) {
            $this->autorecallerRuleRepository = SimpleRepositoryFactory::getRepository(AutorecallerRule::class);
        }
        if (null === $this->scheduledRecallRepository) {
            $this->scheduledRecallRepository = SimpleRepositoryFactory::getRepository(ScheduledRecall::class);
        }
        if (null === $this->scheduledRecallBuilder) {
            $this->scheduledRecallBuilder = ScheduledRecallBuilderFactory::getScheduledRecallBuilder();
        }
    }

    protected function configure()
    {
        $this->setProcessTitle('zruchna.io · Autorecaller');

        $this->addArgument('recall-id', InputArgument::REQUIRED);

        $this->addOption('call-id', null, InputOption::VALUE_OPTIONAL);
        $this->addOption('caller-id', null, InputOption::VALUE_OPTIONAL);
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $recallId = $input->getArgument('recall-id');
        // $callerId = $input->getOption('caller-id');

        $io = new SymfonyStyle($input, $output);

        if (!\is_int($recallId) && !\is_numeric($recallId)) {
            if (!$output->isQuiet()) {
                $io->error('"recall-id" argument must be an integer value');
            }

            return 2;
        }

        $recallId = (int) $recallId;

        $recall = $this->scheduledRecallRepository->find($recallId);
        if (null === $recall) {
            if (!$output->isQuiet()) {
                $io->error('Unable to find provided recall');
            }

            return 1;
        }

        $previousRule = $recall->getAutorecallerRule();
        $nextRule = $this->findNextRule($previousRule);

        if (null === $nextRule) {
            if (!$output->isQuiet()) {
                $io->warning(sprintf('No more rules found for #%d recall', $recallId));
            }

            return 0;
        }

        $nextRecall = $this->scheduledRecallBuilder
            ->reset()
            ->setCallerId($recall->getCallerId())
            ->setRule($nextRule)
            ->getScheduledRecall();

        $this->entityManager->persist($nextRecall);
        $this->entityManager->flush();

        return 0;
    }

    /**
     * @return AutorecallerRule|null
     */
    private function findNextRule(AutorecallerRule $previousRule)
    {
        $recaller = $previousRule->getAutorecaller();
        $previousPriority = $previousRule->getPriority();

        return $this->autorecallerRuleRepository->findNextRule($recaller, $previousPriority);
    }
}
