<?php

namespace Zruchna\FreePbx\Autorecaller\Command;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\StyleInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Zruchna\FreePbx\Autorecaller\Clock\ClockInterface;
use Zruchna\FreePbx\Autorecaller\Contract\Originator;
use Zruchna\FreePbx\Autorecaller\Entity\ScheduledRecall;
use Zruchna\FreePbx\Autorecaller\Factory\ClockFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Doctrine\EntityManagerFactory;
use Zruchna\FreePbx\Autorecaller\Factory\OriginatorFactory;
use Zruchna\FreePbx\Autorecaller\Factory\Repository\SimpleRepositoryFactory;
use Zruchna\FreePbx\Autorecaller\Repository\ScheduledRecallRepository;
use Zruchna\FreePbx\Autorecaller\Service\ActiveCalls;

class OriginateScheduledCallsCommand extends Command
{
    protected static $defaultName = 'originate-scheduled-calls';

    /** @var EntityManagerInterface */
    private $entityManager = null;

    /** @var ScheduledRecallRepository|null */
    private $scheduledRecallRepository = null;

    /** @var ClockInterface|null */
    private $clock = null;

    /** @var Originator|null */
    private $originator = null;

    /** @var ActiveCalls|null */
    private $activeCalls = null;

    /** @var (OutputInterface&StyleInterface)|null */
    private $io = null;

    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->entityManager) {
            $this->entityManager = EntityManagerFactory::getEntityManager();
        }
        if (null === $this->scheduledRecallRepository) {
            $this->scheduledRecallRepository = SimpleRepositoryFactory::getRepository(ScheduledRecall::class);
        }
        if (null === $this->clock) {
            $this->clock = ClockFactory::getClock();
        }
        if (null === $this->originator) {
            $this->originator = OriginatorFactory::getOriginator();
        }
        if (null === $this->activeCalls) {
            $this->activeCalls = new ActiveCalls();
        }
        if (null === $this->io) {
            $this->io = new SymfonyStyle($input, $output);
        }
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $now = $this->clock->now();

        $recalls = $this->scheduledRecallRepository->findActual($now);

        if (!$output->isQuiet()) {
            $this->io->writeln(sprintf('Found %d actual recalls for %s date', \count($recalls), $now->format('Y-m-d H:i:s')));
        }

        foreach ($recalls as $recall) {
            if ($this->activeCalls->isChannelWithPhoneNumberExists($recall->getCallId())) {
                $this->io->note(sprintf('Active channel found with "%s" phone number.', $recall->getCallId()));

                continue;
            }

            $this->originateRecall($recall);
        }

        $this->entityManager->flush();
    }

    /**
     * @return void
     */
    private function originateRecall(ScheduledRecall $recall)
    {
        $rule = $recall->getAutorecallerRule();
        $recaller = $rule->getAutorecaller();

        $callerId = $recall->getCallerId();

        try {
            $accountCode = $this->originator->originateCall(
                $callerId,
                $recaller->getId(),
                $rule->getId(),
                $recall->getId()
            );
        } catch (\Exception $e) {
            $this->io->error(sprintf('Unable to originate recall #%d: %s', $recall->getId(), $e->getMessage()));

            return;
        }

        $recall->setCallId($accountCode);
    }
}
