<?php

namespace FreePBX\Modules;

use Composer\Autoload\ClassLoader;
use FreePBX\Job;
use FreePBX\modules\Autorecaller\OriginateRecallJob;
use FreePBX\modules\Autorecaller\VacuumScheduledCallsTableJob;
use Zruchna\FreePbx\Autorecaller\Factory\Dialplan\DialplanBuilderFactory;

class Autorecaller extends \FreePBX_Helpers implements \BMO
{
    /** @var \FreePBX $FreePBX */
    protected $FreePBX;

    /** @var Job */
    private $jobManagement;

    /** @var ClassLoader */
    private $autoloader;

    /**
     * @param \FreePBX $freepbx
     * @throws \Exception
     * @throws \BadMethodCallException
     */
    public function __construct($freepbx = null)
    {
        parent::__construct($freepbx);

        if (null === $freepbx) {
            throw new \BadMethodCallException('Not given a FreePBX Object');
        }

        $this->FreePBX = $freepbx;
        $this->jobManagement = $freepbx->Job;

        $this->autoloader = require __DIR__.'/src/bootstrap.php';
    }

    /**
     * Set Priority for Dialplan Hooking
     * Core sits at a priority of 600
     * @method myDialplanHooks
     *
     * @return string        Priority
     */
    public static function myDialplanHooks()
    {
        return 910;
    }

    public function install()
    {
        $this->addAgiBinFile('zruchna-autorecaller.php', 'zruchna-autorecaller');

        $this->jobManagement->addClass(
            'autorecaller',
            'originate-recalls',
            OriginateRecallJob::class,
            '* * * * *'
        );

        $this->jobManagement->addClass(
            'autorecaller',
            'vacuum-scheduled-calls-table',
            VacuumScheduledCallsTableJob::class,
            '@daily'
        );
    }

    public function uninstall()
    {
        $this->jobManagement->removeAllByModule('autorecaller');

        $this->removeAgiBinFile('zruchna-autorecaller.php');
    }

    private function addAgiBinFile($filename, $agiName = null)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $srcFilePath = __DIR__.'/agi-bin/'.$filename;
        $dstFilePath = $agibin_dest.'/'.($agiName ?: $filename);

        if (file_exists($dstFilePath)) {
            return;
        }

        try {
            symlink($srcFilePath, $dstFilePath);
        } catch (\ErrorException $e) {
            out('<span class=error>Unable to symlink new agi-bin file: '.$e->getMessage().'</span>');
            out('<p class=mb-0>Link target: <code>'.$srcFilePath.'</code></p>');
            out('<p class=mb-0>Link name: <code>'.$dstFilePath.'</code></p>');
            exit(1);
        }
    }

    private function removeAgiBinFile($filename)
    {
        global $amp_conf;

        $agibin_dest = isset($amp_conf['ASTAGIDIR']) ? $amp_conf['ASTAGIDIR'] : '/var/lib/asterisk/agi-bin';

        $dstFilePath = $agibin_dest.'/'.$filename;

        if (file_exists($dstFilePath)) {
            unlink($dstFilePath);
        }
    }

    public function backup()
    {
        // TODO: Implement backup() method.
    }

    public function restore($backup)
    {
        // TODO: Implement restore() method.
    }

    /**
     * Hook into Dialplan (extensions_additional.conf)
     * @method doDialplanHook
     *
     * @param \extensions $ext The Extensions Class https://wiki.freepbx.org/pages/viewpage.action?pageId=98701336
     * @param string $engine Always Asterisk, Legacy
     * @param string $priority Priority
     */
    public function doDialplanHook($ext, $engine, $priority)
    {
        $builder = DialplanBuilderFactory::getDialplanBuilder();
        $builder->populateExtensions($ext);
    }

    public function doConfigPageInit($page)
    {
    }

    public function getActionBar($request)
    {
        $buttons = array();
        switch ($_GET['display']) {
            case 'autorecaller':
                $buttons = array(
                    'delete' => array(
                        'name' => 'delete',
                        'id' => 'delete',
                        'value' => _('Delete')
                    ),
                    'reset' => array(
                        'name' => 'reset',
                        'id' => 'reset',
                        'value' => _('Reset')
                    ),
                    'submit' => array(
                        'name' => 'submit',
                        'id' => 'submit',
                        'value' => _('Submit')
                    )
                );
                if (empty($_GET['extdisplay'])) {
                    unset($buttons['delete']);
                }
                break;
        }
        return $buttons;
    }

    public function showPage()
    {
        $view = empty($_REQUEST['view']) ? 'main' : $_REQUEST['view'];

        if ('POST' === $_SERVER['REQUEST_METHOD']) {
            $_SERVER['REQUEST_URI'] = preg_replace(
                ['~error(?:=[^&#]*)?~', '~&{2,}~'],
                '&',
                $_SERVER['REQUEST_URI']
            );

            // \needreload();

            header('Location: '.$_SERVER['REQUEST_URI'], true, 302);
            return '';
        }

        return load_view(__DIR__ . '/views/main.php', [
        ]);
    }

    /**
     * @param string $req
     * @param array $setting
     * @return bool
     *
     * @noinspection PhpUnused
     */
    public function ajaxRequest($req, &$setting)
    {
        switch ($req) {
            case 'ping':
                $this->allowAnonymous($setting);

                return true;
            default:
                return false;
        }
    }

    public function ajaxHandler()
    {
        switch ($_REQUEST['command']) {
            case 'ping':
                return 'pong';
            default:
                return false;
        }
    }

    public function getRightNav($request)
    {
        return '';

        return <<<'HTML'
<div class="bootstrap-table">
  <div class="fixed-table-toolbar">
    <div class="bs-bars">
      <div class="toolbar-cbbnav">
        <a href="?display=autorecaller" class="btn btn-default">Main</a>
      </div>
    </div>
  </div>
</div>
HTML;
    }

    /**
     * @return void
     */
    private function allowAnonymous(array &$setting)
    {
        $setting['allowremote'] = true;
        $setting['authenticate'] = false;
    }
}
