<?php

namespace Zruchna\FreePbx\Autoinformer\Repository;

use Doctrine\DBAL\Types\Type;
use Doctrine\ORM\EntityManager;
use Zruchna\FreePbx\Autoinformer\Entity\ScheduledCall;
use Zruchna\FreePbx\Autoinformer\Util\DatetimeUtil;

/**
 * The {@see ScheduledCall} entity repository.
 *
 * @method ScheduledCall|null find($id, $lockMode = null, $lockVersion = null)
 * @method ScheduledCall[]    findBy(array $criteria, array $orderBy = null, $limit = null, $offset = null)
 * @method ScheduledCall|null findOneBy(array $criteria, array $orderBy = null)
 * @method ScheduledCall[]    findAll()
 *
 * @author Stanislau Kviatkouski <7zete7@gmail.com>
 */
class ScheduledCallRepository extends AbstractSimpleRepository
{
    public function __construct(EntityManager $em)
    {
        parent::__construct($em, ScheduledCall::class);
    }

    /**
     * @param int|null $autoinformerId
     * @param int|null $limit
     * @return ScheduledCall[]
     */
    public function findAwaitingCalls(\DateTimeImmutable $atTime, $autoinformerId = null, $limit = null)
    {
        $atTime = DatetimeUtil::toMutableDatetime(DatetimeUtil::toUtcTimezone($atTime));

        $qb = $this->createQueryBuilder('scheduled_calls')
            ->setParameter('at_time', $atTime, Type::DATETIME)
        ;
        $expr = $qb->expr();

        // A Call must not be scheduled, or it must be scheduled before $atTime date
        $qb->andWhere($expr->orX(
            $expr->isNull('scheduled_calls.scheduledAt'),
            $expr->lte('scheduled_calls.scheduledAt', ':at_time')
        ));

        // A Call must not have TTL, or it must be less than $atTime date
        $qb->andWhere($expr->orX(
            $expr->isNull('scheduled_calls.originateUntil'),
            $expr->gt('scheduled_calls.originateUntil', ':at_time')
        ));

        // A Call must not be originated yet
        $qb->andWhere($expr->isNull('scheduled_calls.originatedAt'));

        if (null !== $autoinformerId) {
            $qb
                ->andWhere($expr->eq('scheduled_calls.autoinformer', ':autoinformer_id'))
                ->setParameter('autoinformer_id', $autoinformerId, Type::INTEGER)
            ;
        } else {
            $qb
                ->join('scheduled_calls.autoinformer', 'autoinformers')
                ->andWhere($expr->eq('autoinformers.enabled', 1))
            ;
        }

        if (null !== $limit) {
            $qb->setMaxResults($limit);
        }

        return $qb
            ->getQuery()
            ->getResult();
    }
}
