<?php

namespace Zruchna\FreePbx\Autoinformer\Entity;

use Doctrine\ORM\Mapping as ORM;
use Zruchna\FreePbx\Autoinformer\Repository\ScheduledCallRepository;
use Zruchna\FreePbx\Autoinformer\Util\DatetimeUtil;

/**
 * @ORM\Entity(repositoryClass=ScheduledCallRepository::class)
 * @ORM\Table(name="zruchna_autoinformer_scheduled_calls")
 * @ORM\HasLifecycleCallbacks
 *
 * @phpstan-type TSuperpositionDate of \DateTime|\DateTimeImmutable
 */
class ScheduledCall
{
    /**
     * @ORM\Id
     * @ORM\GeneratedValue(strategy="IDENTITY")
     * @ORM\Column(name="id", type="integer", nullable=false)
     *
     * @var int|null
     */
    private $id = null;

    /**
     * @ORM\ManyToOne(targetEntity=Autoinformer::class)
     * @ORM\JoinColumn(name="autoinformer_id", nullable=false)
     *
     * @var Autoinformer|null
     */
    private $autoinformer = null;

    /**
     * @ORM\Column(name="phone_number", type="string", length=35, nullable=false)
     *
     * @var string|null
     */
    private $phoneNumber = null;

    /**
     * @ORM\Column(name="created_at", type="datetime", nullable=false)
     *
     * @var \DateTime|null
     * @phpstan-var TSuperpositionDate|null
     */
    private $createdAt = null;

    /**
     * @ORM\Column(name="scheduled_at", type="datetime", nullable=true)
     *
     * @var \DateTime|null
     * @phpstan-var TSuperpositionDate|null
     */
    private $scheduledAt = null;

    /**
     * @ORM\Column(name="originated_at", type="datetime", nullable=true)
     *
     * @var \DateTime|null
     * @phpstan-var TSuperpositionDate|null
     */
    private $originatedAt = null;

    /**
     * @ORM\Column(name="originate_until", type="datetime", nullable=true)
     *
     * @var \DateTime|null
     * @phpstan-var TSuperpositionDate|null
     */
    private $originateUntil = null;

    /**
     * @ORM\Column(name="account_code", type="string", length=20, nullable=true)
     *
     * @var string|null
     */
    private $accountCode = null;

    /**
     * @ORM\Column(name="variables", type="json_array", nullable=true)
     *
     * @var array<string, string|string[]>|null
     */
    private $variables;

    /**
     * @return int|null
     */
    public function getId()
    {
        return $this->id;
    }

    /**
     * @param int|null $id
     * @return self
     */
    public function setId($id)
    {
        $this->id = $id;
        return $this;
    }

    /**
     * @return Autoinformer|null
     */
    public function getAutoinformer()
    {
        return $this->autoinformer;
    }

    /**
     * @param Autoinformer|null $autoinformer
     * @return self
     */
    public function setAutoinformer($autoinformer)
    {
        $this->autoinformer = $autoinformer;

        return $this;
    }

    /**
     * @return string|null
     */
    public function getPhoneNumber()
    {
        return $this->phoneNumber;
    }

    /**
     * @param string|null $phoneNumber
     * @return self
     */
    public function setPhoneNumber($phoneNumber)
    {
        $this->phoneNumber = $phoneNumber;

        return $this;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getCreatedAt()
    {
        return DatetimeUtil::toImmutableDatetime($this->createdAt);
    }

    /**
     * @param \DateTimeImmutable|null $createdAt
     * @return self
     */
    public function setCreatedAt($createdAt)
    {
        \assert(null === $createdAt || $createdAt instanceof \DateTimeImmutable);

        $this->createdAt = DatetimeUtil::toMutableDatetime($createdAt);

        return $this;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getScheduledAt()
    {
        return DatetimeUtil::toImmutableDatetime($this->scheduledAt);
    }

    /**
     * @param \DateTimeImmutable|null $scheduledAt
     * @return self
     */
    public function setScheduledAt($scheduledAt)
    {
        \assert(null === $scheduledAt || $scheduledAt instanceof \DateTimeImmutable);

        $this->scheduledAt = DatetimeUtil::toMutableDatetime($scheduledAt);

        return $this;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getOriginatedAt()
    {
        return DatetimeUtil::toImmutableDatetime($this->originatedAt);
    }

    /**
     * @param \DateTimeImmutable|null $originatedAt
     * @return self
     */
    public function setOriginatedAt($originatedAt)
    {
        \assert(null === $originatedAt || $originatedAt instanceof \DateTimeImmutable);

        $this->originatedAt = DatetimeUtil::toMutableDatetime($originatedAt);

        return $this;
    }

    /**
     * @return \DateTimeImmutable|null
     */
    public function getOriginateUntil()
    {
        return DatetimeUtil::toImmutableDatetime($this->originateUntil);
    }

    /**
     * @param \DateTimeImmutable|null $originateUntil
     * @return self
     */
    public function setOriginateUntil($originateUntil)
    {
        \assert(null === $originateUntil || $originateUntil instanceof \DateTimeImmutable);

        $this->originateUntil = DatetimeUtil::toMutableDatetime($originateUntil);

        return $this;
    }

    /**
     * @return string|null
     */
    public function getAccountCode()
    {
        return $this->accountCode;
    }

    /**
     * @param string|null $accountCode
     * @return self
     */
    public function setAccountCode($accountCode)
    {
        $this->accountCode = $accountCode;

        return $this;
    }

    /**
     * @return array<string, string|string[]>|null
     */
    public function getVariables()
    {
        return $this->variables;
    }

    /**
     * @param array<string, string|string[]>|null $variables
     * @return self
     */
    public function setVariables($variables)
    {
        if ([] === $variables) {
            $variables = null;
        }

        $this->variables = $variables;

        return $this;
    }

    /**
     * @ORM\PostLoad
     *
     * @return void
     */
    public function postLoad()
    {
        $this->createdAt = DatetimeUtil::toImmutableDatetime(DatetimeUtil::castAsUtc($this->createdAt));
        $this->scheduledAt = DatetimeUtil::toImmutableDatetime(DatetimeUtil::castAsUtc($this->scheduledAt));
        $this->originatedAt = DatetimeUtil::toImmutableDatetime(DatetimeUtil::castAsUtc($this->originatedAt));
        $this->originateUntil = DatetimeUtil::toImmutableDatetime(DatetimeUtil::castAsUtc($this->originateUntil));
    }

    /**
     * @ORM\PreFlush
     *
     * @return void
     */
    public function preFlush()
    {
        $this->createdAt = DatetimeUtil::toMutableDatetime(DatetimeUtil::toUtcTimezone($this->createdAt));
        $this->scheduledAt = DatetimeUtil::toMutableDatetime(DatetimeUtil::toUtcTimezone($this->scheduledAt));
        $this->originatedAt = DatetimeUtil::toMutableDatetime(DatetimeUtil::toUtcTimezone($this->originatedAt));
        $this->originateUntil = DatetimeUtil::toMutableDatetime(DatetimeUtil::toUtcTimezone($this->originateUntil));
    }
}
