<?php

namespace Zruchna\FreePbx\Autoinformer\Util;

/**
 * @internal
 *
 * @auther Stanislau Kviatkouski <7zete7@gmail.com>
 */
class DatetimeUtil
{
    private static $transportFormat = 'U.u';

    /**
     * @template TDate of \DateTimeInterface|null
     * @phpstan-param TDate $date
     * @phpstan-return (TDate is null ? null : \DateTime)
     */
    public static function toMutableDatetime($date)
    {
        \assert(null === $date || $date instanceof \DateTimeInterface);

        if (null === $date) {
            return $date;
        }
        if ($date instanceof \DateTime) {
            return clone $date;
        }

        return self::createNewInstanceOf(
            \DateTime::class,
            $date,
            $date->getTimezone() ?: null
        );
    }

    /**
     * @template TDate of \DateTimeInterface|null
     * @phpstan-param TDate $date
     * @phpstan-return (TDate is null ? null : \DateTimeImmutable)
     */
    public static function toImmutableDatetime($date)
    {
        \assert(null === $date || $date instanceof \DateTimeInterface);

        if (null === $date) {
            return null;
        }
        if ($date instanceof \DateTimeImmutable) {
            return $date;
        }
        if ($date instanceof \DateTime) {
            return \DateTimeImmutable::createFromMutable($date);
        }

        return self::createNewInstanceOf(
            \DateTimeImmutable::class,
            $date,
            $date->getTimezone() ?: null
        );
    }

    /**
     * @template TDate of \DateTimeInterface|null
     * @phpstan-param TDate $date
     * @phpstan-return TDate
     */
    public static function toUtcTimezone($date)
    {
        \assert(null === $date || $date instanceof \DateTimeInterface);

        if (null === $date) {
            return null;
        }
        if (!($date instanceof \DateTimeImmutable)) {
            $date = clone $date;
        }

        return $date->setTimezone(TimezoneRegistry::getTimezone('UTC'));
    }

    /**
     * @template TDate of \DateTimeInterface|null
     * @param TDate $date
     * @return TDate
     */
    public static function castAsUtc($date)
    {
        \assert(null === $date || $date instanceof \DateTimeInterface);

        if (null === $date) {
            return $date;
        }

        return self::createNewInstanceOf(\get_class($date), $date, TimezoneRegistry::getTimezone('UTC'));
    }

    /**
     * @template TDate of \DateTimeInterface
     * @phpstan-param class-string<TDate> $dateClass
     * @phpstan-return TDate
     */
    protected static function createNewInstanceOf($dateClass, \DateTimeInterface $date, \DateTimeZone $timezone = null)
    {
        \assert(\is_string($dateClass) && \class_exists($dateClass, false));

        if (!\method_exists($dateClass, 'createFromFormat')) {
            throw new \UnexpectedValueException(sprintf('Expected to the %s::createFromFormat() method be exists', $dateClass));
        }

        $formattedDate = $date->format(self::$transportFormat);

        return \call_user_func([$dateClass, 'createFromFormat'], $formattedDate, $timezone);
    }
}
