<?php

namespace Zruchna\FreePbx\Autoinformer\Service;

use Zruchna\FreePbx\Autoinformer\Contract\Originator;
use Zruchna\FreePbx\Autoinformer\Entity\ScheduledCall;

class CallfileOriginator implements Originator
{
    const DEFAULT_CALL_FILE_CONTENT_TEMPLATE = <<<'CALLFILE'
Channel: Local/{caller_id}@zruchna-autoinformer-{autoinformer_id}
Callerid: {caller_id} <{caller_id}>
Set: ZRUCHNA_AUTOINFORMER_ID={autoinformer_id}
Set: ZRUCHNA_AUTOINFORMER_SCHEDULED_CALL_ID={scheduled_call_id}
MaxRetries: 0
RetryTime: 300
WaitTime: 200
Account: {account_code}
Context: ivr-{ivr_id}
Extension: s
Priority: 1
Archive: Yes
CALLFILE;

    const DEFAULT_CALL_FILE_PATH_TEMPLATE = '/var/spool/asterisk/outgoing/{account_code}.call';

    /**
     * @var string
     */
    private $accountCodePrefix;

    /**
     * @var string
     */
    private $callFileContentTemplate;

    /**
     * @var string
     */
    private $callFilePathTemplate;

    public function __construct(
        $accountCodePrefix = 'AD',
        $callFileContentTemplate = self::DEFAULT_CALL_FILE_CONTENT_TEMPLATE,
        $callFilePathTemplate = self::DEFAULT_CALL_FILE_PATH_TEMPLATE
    ) {
        $this->accountCodePrefix = $accountCodePrefix;
        $this->callFileContentTemplate = $callFileContentTemplate;
        $this->callFilePathTemplate = $callFilePathTemplate;
    }

    /**
     * {@inheritDoc}
     */
    public function originateCall(ScheduledCall $scheduledCall)
    {
        $accountCode = uniqid($this->accountCodePrefix);

        $templateEvaluator = $this->createTemplateEvaluator($scheduledCall, $accountCode);

        $callFileContent = $templateEvaluator($this->callFileContentTemplate);
        $callFilePath = $templateEvaluator($this->callFilePathTemplate);

        if (file_exists($callFilePath)) {
            throw new \UnexpectedValueException('Callfile with same name already exists');
        }
        if (false === file_put_contents($callFilePath, $callFileContent, LOCK_EX)) {
            if (isset($php_errormsg) && $php_errormsg) {
                $errorMessageSuffix = ': '.$php_errormsg;
            } else {
                $errorMessageSuffix = '';
            }

            throw new \RuntimeException('Unable to create callfile'.$errorMessageSuffix);
        }

        return $accountCode;
    }

    /**
     * @param string $accountCode
     * @return callable(string $template): string
     */
    protected function createTemplateEvaluator(ScheduledCall $scheduledCall, $accountCode)
    {
        if (null === $autoinformer = $scheduledCall->getAutoinformer()) {
            throw new \UnexpectedValueException('Expected to scheduled call\'s autoinformer be configured');
        }
        if (null === $ivrDetail = $autoinformer->getIvrDetail()) {
            throw new \UnexpectedValueException('Expected to IVR detail of scheduled call\'s autoinformer be configured');
        }
        if (null === $autoinformerId = $autoinformer->getId()) {
            throw new \UnexpectedValueException('Expected to scheduled call\'s autoinformer be contains non-null id value, null provided');
        }
        if (null === $scheduledCallId = $scheduledCall->getId()) {
            throw new \UnexpectedValueException('Expected to scheduled call be contains non-null id value, null provided');
        }
        if (null === $callerId = $scheduledCall->getPhoneNumber()) {
            throw new \UnexpectedValueException('Expected to scheduled call be contains non-null phone number value, null provided');
        }
        if (null === $ivrId = $ivrDetail->getId()) {
            throw new \UnexpectedValueException('Expected to IVR detail of scheduled call\' autoinformer be contains non-null id value, null provided');
        }

        return function ($template) use (
            $autoinformerId,
            $scheduledCallId,
            $accountCode,
            $callerId,
            $ivrId
        ) {
            return strtr($template, array(
                '{autoinformer_id}' => sprintf('%d', $autoinformerId),
                '{scheduled_call_id}' => sprintf('%d', $scheduledCallId),
                '{account_code}' => $accountCode,
                '{caller_id}' => $callerId,
                '{ivr_id}' => $ivrId,
            ));
        };
    }
}
