<?php

namespace Zruchna\FreePbx\Autoinformer\Dialplan\Application;

/**
 * Attempt to connect to another device or endpoint and bridge the call.
 *
 * This application will place calls to one or more specified channels.
 * As soon as one of the requested channels answers,
 * the originating channel will be answered,
 * if it has not already been answered.
 * These two channels will then be active in a bridged call.
 * All other channels that were requested will then be hung up.
 *
 * Unless there is a timeout specified,
 * the Dial application will wait indefinitely until one
 * of the called channels answers, the user hangs up,
 * or if all of the called channels are busy or unavailable.
 * Dialplan executing will continue if no requested channels can be called,
 * or if the timeout expires. This application will report normal termination
 * if the originating channel hangs up, or if the call is bridged and either
 * of the parties in the bridge ends the call.
 *
 * If the `OUTBOUND_GROUP` variable is set, all peer channels created
 * by this application will be put into that group (as in Set(GROUP()=...).
 * If the `OUTBOUND_GROUP_ONCE` variable is set, all peer channels created
 * by this application will be put into that group (as in Set(GROUP()=...).
 * Unlike `OUTBOUND_GROUP`, however, the variable will be unset after use.
 *
 * This application sets the following channel variables:
 * - `DIALEDTIME` - This is the time from dialing a channel until when
 *   it is disconnected.
 * - `ANSWEREDTIME` - This is the amount of time for actual call.
 * - `DIALSTATUS` - This is the status of the call
 * - `CHANUNAVAIL`
 * - `CONGESTION`
 * - `NOANSWER`
 * - `BUSY`
 * - `ANSWER`
 * - `CANCEL`
 * - `DONTCALL` - For the Privacy and Screening Modes.
 *   Will be set if the called party chooses to send the calling party
 *   to the 'Go Away' script.
 * - `TORTURE` - For the Privacy and Screening Modes.
 *   Will be set if the called party chooses to send the calling party
 *   to the 'torture' script.
 * - `INVALIDARGS`
 *
 * @example Dial(Technology/Resource&[Technology2/Resource2[&...]],[timeout,[options,[URL]]])
 *
 * @see https://wiki.asterisk.org/wiki/display/AST/Application_Dial
 */
final class DialApplication extends \ext_dial implements DialplanApplicationInterface
{
    /**
     * @var string[]
     */
    private $destinations;

    /**
     * @var int|null
     */
    private $timeout;

    /**
     * @var string|null
     */
    private $currentOptions;

    /**
     * @var string|null
     */
    private $url;

    /**
     * @param string|string[] $destinations
     * Specification of the device(s) to dial.
     * These must be in the format of Technology/Resource,
     * where Technology represents a particular channel driver,
     * and Resource represents a resource available
     * to that particular channel driver.
     * Optional extra devices to dial in parallel on array value.
     *
     * @param int|null $timeout
     * Specifies the number of seconds we attempt to dial the specified devices.
     * If not specified, this defaults to 136 years.
     *
     * @param string|null $options
     *
     * @param string|null $url
     * The optional URL will be sent to the called party
     * if the channel driver supports it.
     */
    public function __construct($destinations, $timeout = null, $options = null, $url = null)
    {
        $this->destinations = (array) $destinations;
        $this->timeout = $timeout;
        $this->currentOptions = $options;
        $this->url = $url;

        $rawNumber = self::makeParentNumber($this->destinations);

        if (null === $rawOptions = self::makeParentOptions($timeout, $options, $url)) {
            // As it configured on the parent class
            $this->currentOptions = $rawOptions = 'tr';
        }

        parent::__construct($rawNumber, $rawOptions);
    }

    /**
     * @return string[]
     */
    public function getDestinations()
    {
        return $this->destinations;
    }

    /**
     * @return int|null
     */
    public function getTimeout()
    {
        return $this->timeout;
    }

    /**
     * @return string|null
     */
    public function getOptions()
    {
        return $this->currentOptions;
    }

    /**
     * @return string|null
     */
    public function getUrl()
    {
        return $this->url;
    }

    /**
     * @return string
     */
    public function getApplication()
    {
        return 'Dial';
    }

    /**
     * @return string
     */
    public function getData()
    {
        return $this->number.','.$this->options;
    }

    /**
     * @param string[] $destinations
     * @return string
     */
    private static function makeParentNumber(array $destinations)
    {
        return implode('&', $destinations);
    }

    /**
     * @param int|null $timeout
     * @param string|null $options
     * @param string|null $url
     * @return string|null
     */
    private static function makeParentOptions($timeout, $options, $url)
    {
        $parentOptions = array();

        if (null !== $timeout) {
            $parentOptions[] = sprintf('%d', $timeout);
        }

        if (null !== $options) {
            $parentOptions[] = $options;

            if (null !== $url) {
                $parentOptions[] = $url;
            }
        }

        if (!$parentOptions) {
            return null;
        }

        return implode(',', $parentOptions);
    }
}
