<?php

namespace Zruchna\FreePbx\Autoinformer\Command;

use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\StyleInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Zruchna\FreePbx\Autoinformer\Clock\ClockInterface;
use Zruchna\FreePbx\Autoinformer\Contract\Originator;
use Zruchna\FreePbx\Autoinformer\Entity\ScheduledCall;
use Zruchna\FreePbx\Autoinformer\Factory\ClockFactory;
use Zruchna\FreePbx\Autoinformer\Factory\Doctrine\EntityManagerFactory;
use Zruchna\FreePbx\Autoinformer\Factory\OriginatorFactory;
use Zruchna\FreePbx\Autoinformer\Factory\Repository\SimpleRepositoryFactory;
use Zruchna\FreePbx\Autoinformer\Repository\ScheduledCallRepository;
use Zruchna\FreePbx\Autoinformer\Service\ActiveCalls;

class OriginateScheduledCallsCommand extends Command
{
    protected static $defaultName = 'originate-scheduled-calls';

    /** @var EntityManagerInterface */
    private $entityManager = null;

    /** @var ScheduledCallRepository|null */
    private $scheduledCallRepository = null;

    /** @var ClockInterface|null */
    private $clock = null;

    /** @var Originator|null */
    private $originator = null;

    /** @var ActiveCalls|null */
    private $activeCalls = null;

    /** @var (OutputInterface&StyleInterface)|null */
    private $io = null;

    /**
     * @return void
     */
    protected function initialize(InputInterface $input, OutputInterface $output)
    {
        if (null === $this->entityManager) {
            $this->entityManager = EntityManagerFactory::getEntityManager();
        }
        if (null === $this->scheduledCallRepository) {
            $this->scheduledCallRepository = SimpleRepositoryFactory::getRepository(ScheduledCall::class);
        }
        if (null === $this->clock) {
            $this->clock = ClockFactory::getClock();
        }
        if (null === $this->originator) {
            $this->originator = OriginatorFactory::getOriginator();
        }
        if (null === $this->activeCalls) {
            $this->activeCalls = new ActiveCalls();
        }
        if (null === $this->io) {
            $this->io = new SymfonyStyle($input, $output);
        }
    }

    /**
     * @return void
     */
    protected function configure()
    {
        $this->addOption(
            'autoinformer',
            null,
            InputOption::VALUE_REQUIRED,
            'Autorecaller whose scheduled calls must be originated'
        );

        $this->addOption(
            'limit',
            null,
            InputOption::VALUE_REQUIRED,
            'How many scheduled calls must be originated'
        );

        $this->addOption(
            'delay',
            null,
            InputOption::VALUE_REQUIRED,
            'Delay between originates in milliseconds',
            1000
        );
    }

    /**
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        try {
            $autoinformer = $this->resolveAutoinformerInputOptionValue($input);
        } catch (\UnexpectedValueException $e) {
            $this->io->error($e->getMessage());

            return 2;
        }

        try {
            $limit = $this->resolveLimitInputOptionValue($input);
        } catch (\UnexpectedValueException $e) {
            $this->io->error($e->getMessage());

            return 2;
        }

        try {
            $delayMs = $this->resolveDelayInputOptionValue($input);
        } catch (\UnexpectedValueException $e) {
            $this->io->error($e->getMessage());

            return 2;
        }

        $now = $this->clock->now();

        $scheduledCalls = $this->scheduledCallRepository->findAwaitingCalls($now, $autoinformer, $limit);

        if (!$scheduledCalls) {
            if (!$output->isQuiet()) {
                $this->io->note(sprintf('Scheduled calls for %s was not found', $now->format('Y-m-d H:i:s (P)')));
            }

            return 0;
        }

        if (!$output->isQuiet()) {
            $this->io->writeln(sprintf('Found %d scheduled calls for %s date', \count($scheduledCalls), $now->format('Y-m-d H:i:s (P)')));
        }

        $delayUs = $delayMs * 1000;
        $isFirstScheduledCall = true;
        foreach ($scheduledCalls as $scheduledCall) {
            if ($isFirstScheduledCall) {
                $isFirstScheduledCall = false;
            } else {
                $this->io->writeln(sprintf('Delaying next call originating for %s milliseconds...', $delayMs));
                usleep($delayUs);
            }

            if ($this->isActiveCallExists($scheduledCall)) {
                $this->io->note(sprintf('Active channel found with "%s" phone number.', $scheduledCall->getPhoneNumber()));

                continue;
            }

            $this->io->writeln(sprintf('Originating #%d scheduled call...', $scheduledCall->getId()));

            try {
                $accountCode = $this->originateScheduledCall($scheduledCall);
            } catch (\Exception $e) {
                $this->io->warning(sprintf('Failed on #%d scheduled call originating: %s', $scheduledCall->getId(), $e->getMessage()));

                continue;
            } /** @noinspection PhpElementIsNotAvailableInCurrentPhpVersionInspection */ catch (\Throwable $e) {
                $this->io->warning(sprintf('Failed on #%d scheduled call originating: %s', $scheduledCall->getId(), $e->getMessage()));

                continue;
            }

            $this->io->writeln(sprintf('The #%d Scheduled call originated successfully by "%s" account code', $scheduledCall->getId(), $accountCode));

            $scheduledCall
                ->setAccountCode($accountCode)
                ->setOriginatedAt($this->clock->now())
            ;
        }

        $this->entityManager->flush();

        $this->io->success('Scheduled calls originating done');

        return 0;
    }

    /**
     * @return bool
     */
    protected function isActiveCallExists(ScheduledCall $scheduledCall)
    {
        return $this->activeCalls->isChannelWithPhoneNumberExists($scheduledCall->getPhoneNumber());
    }

    /**
     * @throws \RuntimeException
     * @return string
     */
    protected function originateScheduledCall(ScheduledCall $scheduledCall)
    {
        try {
            $accountCode = $this->originator->originateCall($scheduledCall);
        } catch (\Exception $e) {
            throw new \RuntimeException(sprintf('Unable to originate #%d autoinformer call: %s', $scheduledCall->getId(), $e->getMessage()), 0, $e);
        } /** @noinspection PhpElementIsNotAvailableInCurrentPhpVersionInspection */ catch (\Throwable $e) {
            throw new \RuntimeException(sprintf('Unable to originate #%d autoinformer call: %s', $scheduledCall->getId(), $e->getMessage()), 0, $e);
        }

        return $accountCode;
    }

    /**
     * @param string $inputOptionName
     * @throws \UnexpectedValueException
     * @return int|null
     */
    protected function resolveIntegerInputOptionValue(InputInterface $input, $inputOptionName)
    {
        $inputOptionValue = $input->getOption($inputOptionName);

        if (null === $inputOptionValue) {
            return null;
        }
        if (\is_int($inputOptionValue)) {
            return $inputOptionValue;
        }
        if (\is_string($inputOptionValue)) {
            if ($inputOptionValue !== (string) (int) $inputOptionValue) {
                throw new \UnexpectedValueException(sprintf('Unable to understand "%s" %s option value', $inputOptionValue, $inputOptionName));
            }

            return (int) $inputOptionValue;
        }

        throw new \UnexpectedValueException(sprintf('Unsupported type of %s option value provided', $inputOptionName));
    }

    /**
     * @throws \UnexpectedValueException
     * @return int|null
     */
    private function resolveAutoinformerInputOptionValue(InputInterface $input)
    {
        return $this->resolveIntegerInputOptionValue($input, 'autoinformer');
    }

    /**
     * @throws \UnexpectedValueException
     * @return int|null
     */
    private function resolveLimitInputOptionValue(InputInterface $input)
    {
        return $this->resolveIntegerInputOptionValue($input, 'limit');
    }

    /**
     * @throws \UnexpectedValueException
     * @return int|null
     */
    private function resolveDelayInputOptionValue(InputInterface $input)
    {
        return $this->resolveIntegerInputOptionValue($input, 'delay');
    }
}
